// Copyright 2014 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "src/compiler/common-operator.h"
#include "src/compiler/generic-node-inl.h"
#include "src/compiler/graph-inl.h"
#include "src/compiler/js-context-specialization.h"
#include "src/compiler/js-operator.h"
#include "src/compiler/node-aux-data-inl.h"
#include "src/compiler/node-matchers.h"
#include "src/compiler/node-properties-inl.h"

namespace v8 {
namespace internal {
namespace compiler {

// TODO(titzer): factor this out to a common routine with js-typed-lowering.
static void ReplaceEffectfulWithValue(Node* node, Node* value) {
  Node* effect = NULL;
  if (OperatorProperties::HasEffectInput(node->op())) {
    effect = NodeProperties::GetEffectInput(node);
  }

  // Requires distinguishing between value and effect edges.
  UseIter iter = node->uses().begin();
  while (iter != node->uses().end()) {
    if (NodeProperties::IsEffectEdge(iter.edge())) {
      DCHECK_NE(NULL, effect);
      iter = iter.UpdateToAndIncrement(effect);
    } else {
      iter = iter.UpdateToAndIncrement(value);
    }
  }
}


class ContextSpecializationVisitor : public NullNodeVisitor {
 public:
  explicit ContextSpecializationVisitor(JSContextSpecializer* spec)
      : spec_(spec) {}

  GenericGraphVisit::Control Post(Node* node) {
    switch (node->opcode()) {
      case IrOpcode::kJSLoadContext: {
        Reduction r = spec_->ReduceJSLoadContext(node);
        if (r.Changed() && r.replacement() != node) {
          ReplaceEffectfulWithValue(node, r.replacement());
        }
        break;
      }
      case IrOpcode::kJSStoreContext: {
        Reduction r = spec_->ReduceJSStoreContext(node);
        if (r.Changed() && r.replacement() != node) {
          ReplaceEffectfulWithValue(node, r.replacement());
        }
        break;
      }
      default:
        break;
    }
    return GenericGraphVisit::CONTINUE;
  }

 private:
  JSContextSpecializer* spec_;
};


void JSContextSpecializer::SpecializeToContext() {
  ReplaceEffectfulWithValue(context_, jsgraph_->Constant(info_->context()));

  ContextSpecializationVisitor visitor(this);
  jsgraph_->graph()->VisitNodeInputsFromEnd(&visitor);
}


Reduction JSContextSpecializer::ReduceJSLoadContext(Node* node) {
  DCHECK_EQ(IrOpcode::kJSLoadContext, node->opcode());

  ValueMatcher<Handle<Context> > match(NodeProperties::GetValueInput(node, 0));
  // If the context is not constant, no reduction can occur.
  if (!match.HasValue()) {
    return Reducer::NoChange();
  }

  ContextAccess access = OpParameter<ContextAccess>(node);

  // Find the right parent context.
  Context* context = *match.Value();
  for (int i = access.depth(); i > 0; --i) {
    context = context->previous();
  }

  // If the access itself is mutable, only fold-in the parent.
  if (!access.immutable()) {
    // The access does not have to look up a parent, nothing to fold.
    if (access.depth() == 0) {
      return Reducer::NoChange();
    }
    Operator* op = jsgraph_->javascript()->LoadContext(0, access.index(),
                                                       access.immutable());
    node->set_op(op);
    Handle<Object> context_handle = Handle<Object>(context, info_->isolate());
    node->ReplaceInput(0, jsgraph_->Constant(context_handle));
    return Reducer::Changed(node);
  }
  Handle<Object> value =
      Handle<Object>(context->get(access.index()), info_->isolate());

  // Even though the context slot is immutable, the context might have escaped
  // before the function to which it belongs has initialized the slot.
  // We must be conservative and check if the value in the slot is currently the
  // hole or undefined. If it is neither of these, then it must be initialized.
  if (value->IsUndefined() || value->IsTheHole()) {
    return Reducer::NoChange();
  }

  // Success. The context load can be replaced with the constant.
  // TODO(titzer): record the specialization for sharing code across multiple
  // contexts that have the same value in the corresponding context slot.
  return Reducer::Replace(jsgraph_->Constant(value));
}


Reduction JSContextSpecializer::ReduceJSStoreContext(Node* node) {
  DCHECK_EQ(IrOpcode::kJSStoreContext, node->opcode());

  ValueMatcher<Handle<Context> > match(NodeProperties::GetValueInput(node, 0));
  // If the context is not constant, no reduction can occur.
  if (!match.HasValue()) {
    return Reducer::NoChange();
  }

  ContextAccess access = OpParameter<ContextAccess>(node);

  // The access does not have to look up a parent, nothing to fold.
  if (access.depth() == 0) {
    return Reducer::NoChange();
  }

  // Find the right parent context.
  Context* context = *match.Value();
  for (int i = access.depth(); i > 0; --i) {
    context = context->previous();
  }

  Operator* op = jsgraph_->javascript()->StoreContext(0, access.index());
  node->set_op(op);
  Handle<Object> new_context_handle = Handle<Object>(context, info_->isolate());
  node->ReplaceInput(0, jsgraph_->Constant(new_context_handle));

  return Reducer::Changed(node);
}
}
}
}  // namespace v8::internal::compiler
