// Copyright 2007-2008 the V8 project authors. All rights reserved.
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//     * Neither the name of Google Inc. nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// This file was generated at 2008-11-25 16:02:40.592795

#include "unicode-inl.h"
#include <stdlib.h>
#include <stdio.h>

namespace unibrow {

static const int kStartBit = (1 << 30);
static const int kChunkBits = (1 << 15);

/**
 * \file
 * Implementations of functions for working with unicode.
 */

typedef signed short int16_t;  // NOLINT
typedef unsigned short uint16_t;  // NOLINT
typedef int int32_t;  // NOLINT

// All access to the character table should go through this function.
template <int D>
static inline uchar TableGet(const int32_t* table, int index) {
  return table[D * index];
}

static inline uchar GetEntry(int32_t entry) {
  return entry & (kStartBit - 1);
}

static inline bool IsStart(int32_t entry) {
  return (entry & kStartBit) != 0;
}

/**
 * Look up a character in the unicode table using a mix of binary and
 * interpolation search.  For a uniformly distributed array
 * interpolation search beats binary search by a wide margin.  However,
 * in this case interpolation search degenerates because of some very
 * high values in the lower end of the table so this function uses a
 * combination.  The average number of steps to look up the information
 * about a character is around 10, slightly higher if there is no
 * information available about the character.
 */
static bool LookupPredicate(const int32_t* table, uint16_t size, uchar chr) {
  static const int kEntryDist = 1;
  uint16_t value = chr & (kChunkBits - 1);
  unsigned int low = 0;
  unsigned int high = size - 1;
  while (high != low) {
    unsigned int mid = low + ((high - low) >> 1);
    uchar current_value = GetEntry(TableGet<kEntryDist>(table, mid));
    // If we've found an entry less than or equal to this one, and the
    // next one is not also less than this one, we've arrived.
    if ((current_value <= value) &&
        (mid + 1 == size ||
         GetEntry(TableGet<kEntryDist>(table, mid + 1)) > value)) {
      low = mid;
      break;
    } else if (current_value < value) {
      low = mid + 1;
    } else if (current_value > value) {
      // If we've just checked the bottom-most value and it's not
      // the one we're looking for, we're done.
      if (mid == 0) break;
      high = mid - 1;
    }
  }
  int32_t field = TableGet<kEntryDist>(table, low);
  uchar entry = GetEntry(field);
  bool is_start = IsStart(field);
  return (entry == value) ||
          (entry < value && is_start);
}

template <int kW>
struct MultiCharacterSpecialCase {
  uint16_t length;
  uchar chars[kW];
};

// Look up the mapping for the given character in the specified table,
// which is of the specified length and uses the specified special case
// mapping for multi-char mappings.  The next parameter is the character
// following the one to map.  The result will be written in to the result
// buffer and the number of characters written will be returned.  Finally,
// if the allow_caching_ptr is non-null then false will be stored in
// it if the result contains multiple characters or depends on the
// context.
template <int kW>
static int LookupMapping(const int32_t* table,
                         uint16_t size,
                         const MultiCharacterSpecialCase<kW>* multi_chars,
                         uchar chr,
                         uchar next,
                         uchar* result,
                         bool* allow_caching_ptr) {
  static const int kEntryDist = 2;
  uint16_t value = chr & (kChunkBits - 1);
  unsigned int low = 0;
  unsigned int high = size - 1;
  while (high != low) {
    unsigned int mid = low + ((high - low) >> 1);
    uchar current_value = GetEntry(TableGet<kEntryDist>(table, mid));
    // If we've found an entry less than or equal to this one, and the next one
    // is not also less than this one, we've arrived.
    if ((current_value <= value) &&
        (mid + 1 == size ||
         GetEntry(TableGet<kEntryDist>(table, mid + 1)) > value)) {
      low = mid;
      break;
    } else if (current_value < value) {
      low = mid + 1;
    } else if (current_value > value) {
      // If we've just checked the bottom-most value and it's not
      // the one we're looking for, we're done.
      if (mid == 0) break;
      high = mid - 1;
    }
  }
  int32_t field = TableGet<kEntryDist>(table, low);
  uchar entry = GetEntry(field);
  bool is_start = IsStart(field);
  bool found = (entry == value) || (entry < value && is_start);
  if (found) {
    int32_t value = table[2 * low + 1];
    if (value == 0) {
      // 0 means not present
      return 0;
    } else if ((value & 3) == 0) {
      // Low bits 0 means a constant offset from the given character.
      result[0] = chr + (value >> 2);
      return 1;
    } else if ((value & 3) == 1) {
      // Low bits 1 means a special case mapping
      if (allow_caching_ptr) *allow_caching_ptr = false;
      const MultiCharacterSpecialCase<kW>& mapping = multi_chars[value >> 2];
      for (int i = 0; i < mapping.length; i++)
        result[i] = mapping.chars[i];
      return mapping.length;
    } else {
      // Low bits 2 means a really really special case
      if (allow_caching_ptr) *allow_caching_ptr = false;
      // The cases of this switch are defined in unicode.py in the
      // really_special_cases mapping.
      switch (value >> 2) {
        case 1:
          // Really special case 1: upper case sigma.  This letter
          // converts to two different lower case sigmas depending on
          // whether or not it occurs at the end of a word.
          if (next != 0 && Letter::Is(next)) {
            result[0] = 0x03C3;
          } else {
            result[0] = 0x03C2;
          }
          return 1;
        default:
          return 0;
      }
      return -1;
    }
  } else {
    return 0;
  }
}

uchar Utf8::CalculateValue(const byte* str,
                           unsigned length,
                           unsigned* cursor) {
  static const uchar kMaxOneByteChar = 0x7F;
  static const uchar kMaxTwoByteChar = 0x7FF;
  static const uchar kMaxThreeByteChar = 0xFFFF;
  static const uchar kMaxFourByteChar = 0x1FFFFF;

  // We only get called for non-ascii characters.
  if (length == 1) {
    *cursor += 1;
    return kBadChar;
  }
  int first = str[0];
  int second = str[1] ^ 0x80;
  if (second & 0xC0) {
    *cursor += 1;
    return kBadChar;
  }
  if (first < 0xE0) {
    if (first < 0xC0) {
      *cursor += 1;
      return kBadChar;
    }
    uchar l = ((first << 6) | second) & kMaxTwoByteChar;
    if (l <= kMaxOneByteChar) {
      *cursor += 1;
      return kBadChar;
    }
    *cursor += 2;
    return l;
  }
  if (length == 2) {
    *cursor += 1;
    return kBadChar;
  }
  int third = str[2] ^ 0x80;
  if (third & 0xC0) {
    *cursor += 1;
    return kBadChar;
  }
  if (first < 0xF0) {
    uchar l = ((((first << 6) | second) << 6) | third) & kMaxThreeByteChar;
    if (l <= kMaxTwoByteChar) {
      *cursor += 1;
      return kBadChar;
    }
    *cursor += 3;
    return l;
  }
  if (length == 3) {
    *cursor += 1;
    return kBadChar;
  }
  int fourth = str[3] ^ 0x80;
  if (fourth & 0xC0) {
    *cursor += 1;
    return kBadChar;
  }
  if (first < 0xF8) {
    uchar l = (((((first << 6 | second) << 6) | third) << 6) | fourth) &
              kMaxFourByteChar;
    if (l <= kMaxThreeByteChar) {
      *cursor += 1;
      return kBadChar;
    }
    *cursor += 4;
    return l;
  }
  *cursor += 1;
  return kBadChar;
}

const byte* Utf8::ReadBlock(Buffer<const char*> str, byte* buffer,
    unsigned capacity, unsigned* chars_read_ptr, unsigned* offset_ptr) {
  unsigned offset = *offset_ptr;
  // Bail out early if we've reached the end of the string.
  if (offset == str.length()) {
    *chars_read_ptr = 0;
    return NULL;
  }
  const byte* data = reinterpret_cast<const byte*>(str.data());
  if (data[offset] <= kMaxOneByteChar) {
    // The next character is an ascii char so we scan forward over
    // the following ascii characters and return the next pure ascii
    // substring
    const byte* result = data + offset;
    offset++;
    while ((offset < str.length()) && (data[offset] <= kMaxOneByteChar))
      offset++;
    *chars_read_ptr = offset - *offset_ptr;
    *offset_ptr = offset;
    return result;
  } else {
    // The next character is non-ascii so we just fill the buffer
    unsigned cursor = 0;
    unsigned chars_read = 0;
    while (offset < str.length()) {
      uchar c = data[offset];
      if (c <= kMaxOneByteChar) {
        // Fast case for ascii characters
        if (!CharacterStream::EncodeAsciiCharacter(c,
                                                   buffer,
                                                   capacity,
                                                   cursor))
          break;
        offset += 1;
      } else {
        unsigned chars = 0;
        c = Utf8::ValueOf(data + offset, str.length() - offset, &chars);
        if (!CharacterStream::EncodeNonAsciiCharacter(c,
                                                      buffer,
                                                      capacity,
                                                      cursor))
          break;
        offset += chars;
      }
      chars_read++;
    }
    *offset_ptr = offset;
    *chars_read_ptr = chars_read;
    return buffer;
  }
}

unsigned CharacterStream::Length() {
  unsigned result = 0;
  while (has_more()) {
    result++;
    GetNext();
  }
  Rewind();
  return result;
}

void CharacterStream::Seek(unsigned position) {
  Rewind();
  for (unsigned i = 0; i < position; i++) {
    GetNext();
  }
}

// Uppercase:            point.category == 'Lu'

static const uint16_t kUppercaseTable0Size = 509;
static const int32_t kUppercaseTable0[509] = { 1073741889, 90, 1073742016, 214, 1073742040, 222, 256, 258, 260, 262, 264, 266, 268, 270, 272, 274, 276, 278, 280, 282, 284, 286, 288, 290, 292, 294, 296, 298, 300, 302, 304, 306, 308, 310, 313, 315, 317, 319, 321, 323, 325, 327, 330, 332, 334, 336, 338, 340, 342, 344, 346, 348, 350, 352, 354, 356, 358, 360, 362, 364, 366, 368, 370, 372, 374, 1073742200, 377, 379, 381, 1073742209, 386, 388, 1073742214, 391, 1073742217, 395, 1073742222, 401, 1073742227, 404, 1073742230, 408, 1073742236, 413, 1073742239, 416, 418, 420, 1073742246, 423, 425, 428, 1073742254, 431, 1073742257, 435, 437, 1073742263, 440, 444, 452, 455, 458, 461, 463, 465, 467, 469, 471, 473, 475, 478, 480, 482, 484, 486, 488, 490, 492, 494, 497, 500, 1073742326, 504, 506, 508, 510, 512, 514, 516, 518, 520, 522, 524, 526, 528, 530, 532, 534, 536, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556, 558, 560, 562, 1073742394, 571, 1073742397, 574, 577, 1073742403, 582, 584, 586, 588, 590, 902, 1073742728, 906, 908, 1073742734, 911, 1073742737, 929, 1073742755, 939, 1073742802, 980, 984, 986, 988, 990, 992, 994, 996, 998, 1000, 1002, 1004, 1006, 1012, 1015, 1073742841, 1018, 1073742845, 1071, 1120, 1122, 1124, 1126, 1128, 1130, 1132, 1134, 1136, 1138, 1140, 1142, 1144, 1146, 1148, 1150, 1152, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1202, 1204, 1206, 1208, 1210, 1212, 1214, 1073743040, 1217, 1219, 1221, 1223, 1225, 1227, 1229, 1232, 1234, 1236, 1238, 1240, 1242, 1244, 1246, 1248, 1250, 1252, 1254, 1256, 1258, 1260, 1262, 1264, 1266, 1268, 1270, 1272, 1274, 1276, 1278, 1280, 1282, 1284, 1286, 1288, 1290, 1292, 1294, 1296, 1298, 1073743153, 1366, 1073746080, 4293, 7680, 7682, 7684, 7686, 7688, 7690, 7692, 7694, 7696, 7698, 7700, 7702, 7704, 7706, 7708, 7710, 7712, 7714, 7716, 7718, 7720, 7722, 7724, 7726, 7728, 7730, 7732, 7734, 7736, 7738, 7740, 7742, 7744, 7746, 7748, 7750, 7752, 7754, 7756, 7758, 7760, 7762, 7764, 7766, 7768, 7770, 7772, 7774, 7776, 7778, 7780, 7782, 7784, 7786, 7788, 7790, 7792, 7794, 7796, 7798, 7800, 7802, 7804, 7806, 7808, 7810, 7812, 7814, 7816, 7818, 7820, 7822, 7824, 7826, 7828, 7840, 7842, 7844, 7846, 7848, 7850, 7852, 7854, 7856, 7858, 7860, 7862, 7864, 7866, 7868, 7870, 7872, 7874, 7876, 7878, 7880, 7882, 7884, 7886, 7888, 7890, 7892, 7894, 7896, 7898, 7900, 7902, 7904, 7906, 7908, 7910, 7912, 7914, 7916, 7918, 7920, 7922, 7924, 7926, 7928, 1073749768, 7951, 1073749784, 7965, 1073749800, 7983, 1073749816, 7999, 1073749832, 8013, 8025, 8027, 8029, 8031, 1073749864, 8047, 1073749944, 8123, 1073749960, 8139, 1073749976, 8155, 1073749992, 8172, 1073750008, 8187, 8450, 8455, 1073750283, 8461, 1073750288, 8466, 8469, 1073750297, 8477, 8484, 8486, 8488, 1073750314, 8493, 1073750320, 8499, 1073750334, 8511, 8517, 8579, 1073753088, 11310, 11360, 1073753186, 11364, 11367, 11369, 11371, 11381, 11392, 11394, 11396, 11398, 11400, 11402, 11404, 11406, 11408, 11410, 11412, 11414, 11416, 11418, 11420, 11422, 11424, 11426, 11428, 11430, 11432, 11434, 11436, 11438, 11440, 11442, 11444, 11446, 11448, 11450, 11452, 11454, 11456, 11458, 11460, 11462, 11464, 11466, 11468, 11470, 11472, 11474, 11476, 11478, 11480, 11482, 11484, 11486, 11488, 11490 }; // NOLINT
static const uint16_t kUppercaseTable1Size = 2;
static const int32_t kUppercaseTable1[2] = { 1073774369, 32570 }; // NOLINT
static const uint16_t kUppercaseTable2Size = 2;
static const int32_t kUppercaseTable2[2] = { 1073742848, 1063 }; // NOLINT
static const uint16_t kUppercaseTable3Size = 58;
static const int32_t kUppercaseTable3[58] = { 1073763328, 21529, 1073763380, 21581, 1073763432, 21633, 21660, 1073763486, 21663, 21666, 1073763493, 21670, 1073763497, 21676, 1073763502, 21685, 1073763536, 21737, 1073763588, 21765, 1073763591, 21770, 1073763597, 21780, 1073763606, 21788, 1073763640, 21817, 1073763643, 21822, 1073763648, 21828, 21830, 1073763658, 21840, 1073763692, 21893, 1073763744, 21945, 1073763796, 21997, 1073763848, 22049, 1073763900, 22101, 1073763952, 22153, 1073764008, 22208, 1073764066, 22266, 1073764124, 22324, 1073764182, 22382, 1073764240, 22440, 22474 }; // NOLINT
bool Uppercase::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kUppercaseTable0,
                                       kUppercaseTable0Size,
                                       c);
    case 1: return LookupPredicate(kUppercaseTable1,
                                       kUppercaseTable1Size,
                                       c);
    case 2: return LookupPredicate(kUppercaseTable2,
                                       kUppercaseTable2Size,
                                       c);
    case 3: return LookupPredicate(kUppercaseTable3,
                                       kUppercaseTable3Size,
                                       c);
    default: return false;
  }
}

// Lowercase:            point.category == 'Ll'

static const uint16_t kLowercaseTable0Size = 528;
static const int32_t kLowercaseTable0[528] = { 1073741921, 122, 170, 181, 186, 1073742047, 246, 1073742072, 255, 257, 259, 261, 263, 265, 267, 269, 271, 273, 275, 277, 279, 281, 283, 285, 287, 289, 291, 293, 295, 297, 299, 301, 303, 305, 307, 309, 1073742135, 312, 314, 316, 318, 320, 322, 324, 326, 1073742152, 329, 331, 333, 335, 337, 339, 341, 343, 345, 347, 349, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 378, 380, 1073742206, 384, 387, 389, 392, 1073742220, 397, 402, 405, 1073742233, 411, 414, 417, 419, 421, 424, 1073742250, 427, 429, 432, 436, 438, 1073742265, 442, 1073742269, 447, 454, 457, 460, 462, 464, 466, 468, 470, 472, 474, 1073742300, 477, 479, 481, 483, 485, 487, 489, 491, 493, 1073742319, 496, 499, 501, 505, 507, 509, 511, 513, 515, 517, 519, 521, 523, 525, 527, 529, 531, 533, 535, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 1073742387, 569, 572, 1073742399, 576, 578, 583, 585, 587, 589, 1073742415, 659, 1073742485, 687, 1073742715, 893, 912, 1073742764, 974, 1073742800, 977, 1073742805, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1001, 1003, 1005, 1073742831, 1011, 1013, 1016, 1073742843, 1020, 1073742896, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1135, 1137, 1139, 1141, 1143, 1145, 1147, 1149, 1151, 1153, 1163, 1165, 1167, 1169, 1171, 1173, 1175, 1177, 1179, 1181, 1183, 1185, 1187, 1189, 1191, 1193, 1195, 1197, 1199, 1201, 1203, 1205, 1207, 1209, 1211, 1213, 1215, 1218, 1220, 1222, 1224, 1226, 1228, 1073743054, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1269, 1271, 1273, 1275, 1277, 1279, 1281, 1283, 1285, 1287, 1289, 1291, 1293, 1295, 1297, 1299, 1073743201, 1415, 1073749248, 7467, 1073749346, 7543, 1073749369, 7578, 7681, 7683, 7685, 7687, 7689, 7691, 7693, 7695, 7697, 7699, 7701, 7703, 7705, 7707, 7709, 7711, 7713, 7715, 7717, 7719, 7721, 7723, 7725, 7727, 7729, 7731, 7733, 7735, 7737, 7739, 7741, 7743, 7745, 7747, 7749, 7751, 7753, 7755, 7757, 7759, 7761, 7763, 7765, 7767, 7769, 7771, 7773, 7775, 7777, 7779, 7781, 7783, 7785, 7787, 7789, 7791, 7793, 7795, 7797, 7799, 7801, 7803, 7805, 7807, 7809, 7811, 7813, 7815, 7817, 7819, 7821, 7823, 7825, 7827, 1073749653, 7835, 7841, 7843, 7845, 7847, 7849, 7851, 7853, 7855, 7857, 7859, 7861, 7863, 7865, 7867, 7869, 7871, 7873, 7875, 7877, 7879, 7881, 7883, 7885, 7887, 7889, 7891, 7893, 7895, 7897, 7899, 7901, 7903, 7905, 7907, 7909, 7911, 7913, 7915, 7917, 7919, 7921, 7923, 7925, 7927, 7929, 1073749760, 7943, 1073749776, 7957, 1073749792, 7975, 1073749808, 7991, 1073749824, 8005, 1073749840, 8023, 1073749856, 8039, 1073749872, 8061, 1073749888, 8071, 1073749904, 8087, 1073749920, 8103, 1073749936, 8116, 1073749942, 8119, 8126, 1073749954, 8132, 1073749958, 8135, 1073749968, 8147, 1073749974, 8151, 1073749984, 8167, 1073750002, 8180, 1073750006, 8183, 8305, 8319, 8458, 1073750286, 8463, 8467, 8495, 8500, 8505, 1073750332, 8509, 1073750342, 8521, 8526, 8580, 1073753136, 11358, 11361, 1073753189, 11366, 11368, 11370, 11372, 11380, 1073753206, 11383, 11393, 11395, 11397, 11399, 11401, 11403, 11405, 11407, 11409, 11411, 11413, 11415, 11417, 11419, 11421, 11423, 11425, 11427, 11429, 11431, 11433, 11435, 11437, 11439, 11441, 11443, 11445, 11447, 11449, 11451, 11453, 11455, 11457, 11459, 11461, 11463, 11465, 11467, 11469, 11471, 11473, 11475, 11477, 11479, 11481, 11483, 11485, 11487, 11489, 1073753315, 11492, 1073753344, 11557 }; // NOLINT
static const uint16_t kLowercaseTable1Size = 6;
static const int32_t kLowercaseTable1[6] = { 1073773312, 31494, 1073773331, 31511, 1073774401, 32602 }; // NOLINT
static const uint16_t kLowercaseTable2Size = 2;
static const int32_t kLowercaseTable2[2] = { 1073742888, 1103 }; // NOLINT
static const uint16_t kLowercaseTable3Size = 54;
static const int32_t kLowercaseTable3[54] = { 1073763354, 21555, 1073763406, 21588, 1073763414, 21607, 1073763458, 21659, 1073763510, 21689, 21691, 1073763517, 21699, 1073763525, 21711, 1073763562, 21763, 1073763614, 21815, 1073763666, 21867, 1073763718, 21919, 1073763770, 21971, 1073763822, 22023, 1073763874, 22075, 1073763926, 22127, 1073763978, 22181, 1073764034, 22234, 1073764060, 22241, 1073764092, 22292, 1073764118, 22299, 1073764150, 22350, 1073764176, 22357, 1073764208, 22408, 1073764234, 22415, 1073764266, 22466, 1073764292, 22473, 22475 }; // NOLINT
bool Lowercase::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kLowercaseTable0,
                                       kLowercaseTable0Size,
                                       c);
    case 1: return LookupPredicate(kLowercaseTable1,
                                       kLowercaseTable1Size,
                                       c);
    case 2: return LookupPredicate(kLowercaseTable2,
                                       kLowercaseTable2Size,
                                       c);
    case 3: return LookupPredicate(kLowercaseTable3,
                                       kLowercaseTable3Size,
                                       c);
    default: return false;
  }
}

// Letter:               point.category in ['Lu', 'Ll', 'Lt', 'Lm', 'Lo' ]

static const uint16_t kLetterTable0Size = 476;
static const int32_t kLetterTable0[476] = { 1073741889, 90, 1073741921, 122, 170, 181, 186, 1073742016, 214, 1073742040, 246, 1073742072, 705, 1073742534, 721, 1073742560, 740, 750, 1073742714, 893, 902, 1073742728, 906, 908, 1073742734, 929, 1073742755, 974, 1073742800, 1013, 1073742839, 1153, 1073742986, 1299, 1073743153, 1366, 1369, 1073743201, 1415, 1073743312, 1514, 1073743344, 1522, 1073743393, 1594, 1073743424, 1610, 1073743470, 1647, 1073743473, 1747, 1749, 1073743589, 1766, 1073743598, 1775, 1073743610, 1788, 1791, 1808, 1073743634, 1839, 1073743693, 1901, 1073743744, 1957, 1969, 1073743818, 2026, 1073743860, 2037, 2042, 1073744132, 2361, 2365, 2384, 1073744216, 2401, 1073744251, 2431, 1073744261, 2444, 1073744271, 2448, 1073744275, 2472, 1073744298, 2480, 2482, 1073744310, 2489, 2493, 2510, 1073744348, 2525, 1073744351, 2529, 1073744368, 2545, 1073744389, 2570, 1073744399, 2576, 1073744403, 2600, 1073744426, 2608, 1073744434, 2611, 1073744437, 2614, 1073744440, 2617, 1073744473, 2652, 2654, 1073744498, 2676, 1073744517, 2701, 1073744527, 2705, 1073744531, 2728, 1073744554, 2736, 1073744562, 2739, 1073744565, 2745, 2749, 2768, 1073744608, 2785, 1073744645, 2828, 1073744655, 2832, 1073744659, 2856, 1073744682, 2864, 1073744690, 2867, 1073744693, 2873, 2877, 1073744732, 2909, 1073744735, 2913, 2929, 2947, 1073744773, 2954, 1073744782, 2960, 1073744786, 2965, 1073744793, 2970, 2972, 1073744798, 2975, 1073744803, 2980, 1073744808, 2986, 1073744814, 3001, 1073744901, 3084, 1073744910, 3088, 1073744914, 3112, 1073744938, 3123, 1073744949, 3129, 1073744992, 3169, 1073745029, 3212, 1073745038, 3216, 1073745042, 3240, 1073745066, 3251, 1073745077, 3257, 3261, 3294, 1073745120, 3297, 1073745157, 3340, 1073745166, 3344, 1073745170, 3368, 1073745194, 3385, 1073745248, 3425, 1073745285, 3478, 1073745306, 3505, 1073745331, 3515, 3517, 1073745344, 3526, 1073745409, 3632, 1073745458, 3635, 1073745472, 3654, 1073745537, 3714, 3716, 1073745543, 3720, 3722, 3725, 1073745556, 3735, 1073745561, 3743, 1073745569, 3747, 3749, 3751, 1073745578, 3755, 1073745581, 3760, 1073745586, 3763, 3773, 1073745600, 3780, 3782, 1073745628, 3805, 3840, 1073745728, 3911, 1073745737, 3946, 1073745800, 3979, 1073745920, 4129, 1073745955, 4135, 1073745961, 4138, 1073746000, 4181, 1073746080, 4293, 1073746128, 4346, 4348, 1073746176, 4441, 1073746271, 4514, 1073746344, 4601, 1073746432, 4680, 1073746506, 4685, 1073746512, 4694, 4696, 1073746522, 4701, 1073746528, 4744, 1073746570, 4749, 1073746576, 4784, 1073746610, 4789, 1073746616, 4798, 4800, 1073746626, 4805, 1073746632, 4822, 1073746648, 4880, 1073746706, 4885, 1073746712, 4954, 1073746816, 5007, 1073746848, 5108, 1073746945, 5740, 1073747567, 5750, 1073747585, 5786, 1073747616, 5866, 1073747712, 5900, 1073747726, 5905, 1073747744, 5937, 1073747776, 5969, 1073747808, 5996, 1073747822, 6000, 1073747840, 6067, 6103, 6108, 1073748000, 6263, 1073748096, 6312, 1073748224, 6428, 1073748304, 6509, 1073748336, 6516, 1073748352, 6569, 1073748417, 6599, 1073748480, 6678, 1073748741, 6963, 1073748805, 6987, 1073749248, 7615, 1073749504, 7835, 1073749664, 7929, 1073749760, 7957, 1073749784, 7965, 1073749792, 8005, 1073749832, 8013, 1073749840, 8023, 8025, 8027, 8029, 1073749855, 8061, 1073749888, 8116, 1073749942, 8124, 8126, 1073749954, 8132, 1073749958, 8140, 1073749968, 8147, 1073749974, 8155, 1073749984, 8172, 1073750002, 8180, 1073750006, 8188, 8305, 8319, 1073750160, 8340, 8450, 8455, 1073750282, 8467, 8469, 1073750297, 8477, 8484, 8486, 8488, 1073750314, 8493, 1073750319, 8505, 1073750332, 8511, 1073750341, 8521, 8526, 1073750403, 8580, 1073753088, 11310, 1073753136, 11358, 1073753184, 11372, 1073753204, 11383, 1073753216, 11492, 1073753344, 11557, 1073753392, 11621, 11631, 1073753472, 11670, 1073753504, 11686, 1073753512, 11694, 1073753520, 11702, 1073753528, 11710, 1073753536, 11718, 1073753544, 11726, 1073753552, 11734, 1073753560, 11742, 1073754117, 12294, 1073754161, 12341, 1073754171, 12348, 1073754177, 12438, 1073754269, 12447, 1073754273, 12538, 1073754364, 12543, 1073754373, 12588, 1073754417, 12686, 1073754528, 12727, 1073754608, 12799, 1073755136, 19893, 1073761792, 32767 }; // NOLINT
static const uint16_t kLetterTable1Size = 68;
static const int32_t kLetterTable1[68] = { 1073741824, 8123, 1073750016, 9356, 1073751831, 10010, 1073752064, 10241, 1073752067, 10245, 1073752071, 10250, 1073752076, 10274, 1073752128, 10355, 1073753088, 22435, 1073772800, 31277, 1073773104, 31338, 1073773168, 31449, 1073773312, 31494, 1073773331, 31511, 31517, 1073773343, 31528, 1073773354, 31542, 1073773368, 31548, 31550, 1073773376, 31553, 1073773379, 31556, 1073773382, 31665, 1073773523, 32061, 1073773904, 32143, 1073773970, 32199, 1073774064, 32251, 1073774192, 32372, 1073774198, 32508, 1073774369, 32570, 1073774401, 32602, 1073774438, 32702, 1073774530, 32711, 1073774538, 32719, 1073774546, 32727, 1073774554, 32732 }; // NOLINT
static const uint16_t kLetterTable2Size = 48;
static const int32_t kLetterTable2[48] = { 1073741824, 11, 1073741837, 38, 1073741864, 58, 1073741884, 61, 1073741887, 77, 1073741904, 93, 1073741952, 250, 1073742592, 798, 1073742640, 832, 1073742658, 841, 1073742720, 925, 1073742752, 963, 1073742792, 975, 1073742848, 1181, 1073743872, 2053, 2056, 1073743882, 2101, 1073743927, 2104, 2108, 2111, 1073744128, 2325, 2560, 1073744400, 2579, 1073744405, 2583, 1073744409, 2611, 1073750016, 9070 }; // NOLINT
static const uint16_t kLetterTable3Size = 57;
static const int32_t kLetterTable3[57] = { 1073763328, 21588, 1073763414, 21660, 1073763486, 21663, 21666, 1073763493, 21670, 1073763497, 21676, 1073763502, 21689, 21691, 1073763517, 21699, 1073763525, 21765, 1073763591, 21770, 1073763597, 21780, 1073763606, 21788, 1073763614, 21817, 1073763643, 21822, 1073763648, 21828, 21830, 1073763658, 21840, 1073763666, 22181, 1073764008, 22208, 1073764034, 22234, 1073764060, 22266, 1073764092, 22292, 1073764118, 22324, 1073764150, 22350, 1073764176, 22382, 1073764208, 22408, 1073764234, 22440, 1073764266, 22466, 1073764292, 22475 }; // NOLINT
static const uint16_t kLetterTable4Size = 2;
static const int32_t kLetterTable4[2] = { 1073741824, 32767 }; // NOLINT
static const uint16_t kLetterTable5Size = 4;
static const int32_t kLetterTable5[4] = { 1073741824, 9942, 1073772544, 31261 }; // NOLINT
bool Letter::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kLetterTable0,
                                       kLetterTable0Size,
                                       c);
    case 1: return LookupPredicate(kLetterTable1,
                                       kLetterTable1Size,
                                       c);
    case 2: return LookupPredicate(kLetterTable2,
                                       kLetterTable2Size,
                                       c);
    case 3: return LookupPredicate(kLetterTable3,
                                       kLetterTable3Size,
                                       c);
    case 4: return LookupPredicate(kLetterTable4,
                                       kLetterTable4Size,
                                       c);
    case 5: return LookupPredicate(kLetterTable5,
                                       kLetterTable5Size,
                                       c);
    default: return false;
  }
}

// Space:                point.category == 'Zs'

static const uint16_t kSpaceTable0Size = 9;
static const int32_t kSpaceTable0[9] = { 32, 160, 5760, 6158, 1073750016, 8202, 8239, 8287, 12288 }; // NOLINT
bool Space::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kSpaceTable0,
                                       kSpaceTable0Size,
                                       c);
    default: return false;
  }
}

// Number:               point.category in ['Nd', 'Nl', 'No' ]

static const uint16_t kNumberTable0Size = 86;
static const int32_t kNumberTable0[86] = { 1073741872, 57, 1073742002, 179, 185, 1073742012, 190, 1073743456, 1641, 1073743600, 1785, 1073743808, 1993, 1073744230, 2415, 1073744358, 2543, 1073744372, 2553, 1073744486, 2671, 1073744614, 2799, 1073744742, 2927, 1073744870, 3058, 1073744998, 3183, 1073745126, 3311, 1073745254, 3439, 1073745488, 3673, 1073745616, 3801, 1073745696, 3891, 1073745984, 4169, 1073746793, 4988, 1073747694, 5872, 1073747936, 6121, 1073747952, 6137, 1073747984, 6169, 1073748294, 6479, 1073748432, 6617, 1073748816, 7001, 8304, 1073750132, 8313, 1073750144, 8329, 1073750355, 8578, 1073751136, 9371, 1073751274, 9471, 1073751926, 10131, 11517, 12295, 1073754145, 12329, 1073754168, 12346, 1073754514, 12693, 1073754656, 12841, 1073754705, 12895, 1073754752, 12937, 1073754801, 12991 }; // NOLINT
static const uint16_t kNumberTable1Size = 2;
static const int32_t kNumberTable1[2] = { 1073774352, 32537 }; // NOLINT
static const uint16_t kNumberTable2Size = 19;
static const int32_t kNumberTable2[19] = { 1073742087, 307, 1073742144, 376, 394, 1073742624, 803, 833, 842, 1073742801, 981, 1073743008, 1193, 1073744150, 2329, 1073744448, 2631, 1073751040, 9314 }; // NOLINT
static const uint16_t kNumberTable3Size = 4;
static const int32_t kNumberTable3[4] = { 1073763168, 21361, 1073764302, 22527 }; // NOLINT
bool Number::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kNumberTable0,
                                       kNumberTable0Size,
                                       c);
    case 1: return LookupPredicate(kNumberTable1,
                                       kNumberTable1Size,
                                       c);
    case 2: return LookupPredicate(kNumberTable2,
                                       kNumberTable2Size,
                                       c);
    case 3: return LookupPredicate(kNumberTable3,
                                       kNumberTable3Size,
                                       c);
    default: return false;
  }
}

// WhiteSpace:           'Ws' in point.properties

static const uint16_t kWhiteSpaceTable0Size = 14;
static const int32_t kWhiteSpaceTable0[14] = { 1073741833, 13, 32, 133, 160, 5760, 6158, 1073750016, 8202, 1073750056, 8233, 8239, 8287, 12288 }; // NOLINT
bool WhiteSpace::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kWhiteSpaceTable0,
                                       kWhiteSpaceTable0Size,
                                       c);
    default: return false;
  }
}

// LineTerminator:       'Lt' in point.properties

static const uint16_t kLineTerminatorTable0Size = 4;
static const int32_t kLineTerminatorTable0[4] = { 10, 13, 1073750056, 8233 }; // NOLINT
bool LineTerminator::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kLineTerminatorTable0,
                                       kLineTerminatorTable0Size,
                                       c);
    default: return false;
  }
}

// CombiningMark:        point.category in ['Mn', 'Mc']

static const uint16_t kCombiningMarkTable0Size = 214;
static const int32_t kCombiningMarkTable0[214] = { 1073742592, 879, 1073742979, 1158, 1073743249, 1469, 1471, 1073743297, 1474, 1073743300, 1477, 1479, 1073743376, 1557, 1073743435, 1630, 1648, 1073743574, 1756, 1073743583, 1764, 1073743591, 1768, 1073743594, 1773, 1809, 1073743664, 1866, 1073743782, 1968, 1073743851, 2035, 1073744129, 2307, 2364, 1073744190, 2381, 1073744209, 2388, 1073744226, 2403, 1073744257, 2435, 2492, 1073744318, 2500, 1073744327, 2504, 1073744331, 2509, 2519, 1073744354, 2531, 1073744385, 2563, 2620, 1073744446, 2626, 1073744455, 2632, 1073744459, 2637, 1073744496, 2673, 1073744513, 2691, 2748, 1073744574, 2757, 1073744583, 2761, 1073744587, 2765, 1073744610, 2787, 1073744641, 2819, 2876, 1073744702, 2883, 1073744711, 2888, 1073744715, 2893, 1073744726, 2903, 2946, 1073744830, 3010, 1073744838, 3016, 1073744842, 3021, 3031, 1073744897, 3075, 1073744958, 3140, 1073744966, 3144, 1073744970, 3149, 1073744981, 3158, 1073745026, 3203, 3260, 1073745086, 3268, 1073745094, 3272, 1073745098, 3277, 1073745109, 3286, 1073745122, 3299, 1073745154, 3331, 1073745214, 3395, 1073745222, 3400, 1073745226, 3405, 3415, 1073745282, 3459, 3530, 1073745359, 3540, 3542, 1073745368, 3551, 1073745394, 3571, 3633, 1073745460, 3642, 1073745479, 3662, 3761, 1073745588, 3769, 1073745595, 3772, 1073745608, 3789, 1073745688, 3865, 3893, 3895, 3897, 1073745726, 3903, 1073745777, 3972, 1073745798, 3975, 1073745808, 3991, 1073745817, 4028, 4038, 1073745964, 4146, 1073745974, 4153, 1073746006, 4185, 4959, 1073747730, 5908, 1073747762, 5940, 1073747794, 5971, 1073747826, 6003, 1073747894, 6099, 6109, 1073747979, 6157, 6313, 1073748256, 6443, 1073748272, 6459, 1073748400, 6592, 1073748424, 6601, 1073748503, 6683, 1073748736, 6916, 1073748788, 6980, 1073748843, 7027, 1073749440, 7626, 1073749502, 7679, 1073750224, 8412, 8417, 1073750245, 8431, 1073754154, 12335, 1073754265, 12442 }; // NOLINT
static const uint16_t kCombiningMarkTable1Size = 10;
static const int32_t kCombiningMarkTable1[10] = { 10242, 10246, 10251, 1073752099, 10279, 31518, 1073774080, 32271, 1073774112, 32291 }; // NOLINT
static const uint16_t kCombiningMarkTable2Size = 9;
static const int32_t kCombiningMarkTable2[9] = { 1073744385, 2563, 1073744389, 2566, 1073744396, 2575, 1073744440, 2618, 2623 }; // NOLINT
static const uint16_t kCombiningMarkTable3Size = 12;
static const int32_t kCombiningMarkTable3[12] = { 1073762661, 20841, 1073762669, 20850, 1073762683, 20866, 1073762693, 20875, 1073762730, 20909, 1073762882, 21060 }; // NOLINT
static const uint16_t kCombiningMarkTable28Size = 2;
static const int32_t kCombiningMarkTable28[2] = { 1073742080, 495 }; // NOLINT
bool CombiningMark::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kCombiningMarkTable0,
                                       kCombiningMarkTable0Size,
                                       c);
    case 1: return LookupPredicate(kCombiningMarkTable1,
                                       kCombiningMarkTable1Size,
                                       c);
    case 2: return LookupPredicate(kCombiningMarkTable2,
                                       kCombiningMarkTable2Size,
                                       c);
    case 3: return LookupPredicate(kCombiningMarkTable3,
                                       kCombiningMarkTable3Size,
                                       c);
    case 28: return LookupPredicate(kCombiningMarkTable28,
                                       kCombiningMarkTable28Size,
                                       c);
    default: return false;
  }
}

// ConnectorPunctuation: point.category == 'Pc'

static const uint16_t kConnectorPunctuationTable0Size = 4;
static const int32_t kConnectorPunctuationTable0[4] = { 95, 1073750079, 8256, 8276 }; // NOLINT
static const uint16_t kConnectorPunctuationTable1Size = 5;
static const int32_t kConnectorPunctuationTable1[5] = { 1073774131, 32308, 1073774157, 32335, 32575 }; // NOLINT
bool ConnectorPunctuation::Is(uchar c) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupPredicate(kConnectorPunctuationTable0,
                                       kConnectorPunctuationTable0Size,
                                       c);
    case 1: return LookupPredicate(kConnectorPunctuationTable1,
                                       kConnectorPunctuationTable1Size,
                                       c);
    default: return false;
  }
}

static const MultiCharacterSpecialCase<3> kToLowercaseMultiStrings0[] = { {2, {105, 775}}, {0, {0}} }; // NOLINT
static const uint16_t kToLowercaseTable0Size = 531;
static const int32_t kToLowercaseTable0[1062] = { 1073741889, 128, 90, 128, 1073742016, 128, 214, 128, 1073742040, 128, 222, 128, 256, 4, 258, 4, 260, 4, 262, 4, 264, 4, 266, 4, 268, 4, 270, 4, 272, 4, 274, 4, 276, 4, 278, 4, 280, 4, 282, 4, 284, 4, 286, 4, 288, 4, 290, 4, 292, 4, 294, 4, 296, 4, 298, 4, 300, 4, 302, 4, 304, 1, 306, 4, 308, 4, 310, 4, 313, 4, 315, 4, 317, 4, 319, 4, 321, 4, 323, 4, 325, 4, 327, 4, 330, 4, 332, 4, 334, 4, 336, 4, 338, 4, 340, 4, 342, 4, 344, 4, 346, 4, 348, 4, 350, 4, 352, 4, 354, 4, 356, 4, 358, 4, 360, 4, 362, 4, 364, 4, 366, 4, 368, 4, 370, 4, 372, 4, 374, 4, 376, -484, 377, 4, 379, 4, 381, 4, 385, 840, 386, 4, 388, 4, 390, 824, 391, 4, 1073742217, 820, 394, 820, 395, 4, 398, 316, 399, 808, 400, 812, 401, 4, 403, 820, 404, 828, 406, 844, 407, 836, 408, 4, 412, 844, 413, 852, 415, 856, 416, 4, 418, 4, 420, 4, 422, 872, 423, 4, 425, 872, 428, 4, 430, 872, 431, 4, 1073742257, 868, 434, 868, 435, 4, 437, 4, 439, 876, 440, 4, 444, 4, 452, 8, 453, 4, 455, 8, 456, 4, 458, 8, 459, 4, 461, 4, 463, 4, 465, 4, 467, 4, 469, 4, 471, 4, 473, 4, 475, 4, 478, 4, 480, 4, 482, 4, 484, 4, 486, 4, 488, 4, 490, 4, 492, 4, 494, 4, 497, 8, 498, 4, 500, 4, 502, -388, 503, -224, 504, 4, 506, 4, 508, 4, 510, 4, 512, 4, 514, 4, 516, 4, 518, 4, 520, 4, 522, 4, 524, 4, 526, 4, 528, 4, 530, 4, 532, 4, 534, 4, 536, 4, 538, 4, 540, 4, 542, 4, 544, -520, 546, 4, 548, 4, 550, 4, 552, 4, 554, 4, 556, 4, 558, 4, 560, 4, 562, 4, 570, 43180, 571, 4, 573, -652, 574, 43168, 577, 4, 579, -780, 580, 276, 581, 284, 582, 4, 584, 4, 586, 4, 588, 4, 590, 4, 902, 152, 1073742728, 148, 906, 148, 908, 256, 1073742734, 252, 911, 252, 1073742737, 128, 929, 128, 1073742755, 6, 939, 128, 984, 4, 986, 4, 988, 4, 990, 4, 992, 4, 994, 4, 996, 4, 998, 4, 1000, 4, 1002, 4, 1004, 4, 1006, 4, 1012, -240, 1015, 4, 1017, -28, 1018, 4, 1073742845, -520, 1023, -520, 1073742848, 320, 1039, 320, 1073742864, 128, 1071, 128, 1120, 4, 1122, 4, 1124, 4, 1126, 4, 1128, 4, 1130, 4, 1132, 4, 1134, 4, 1136, 4, 1138, 4, 1140, 4, 1142, 4, 1144, 4, 1146, 4, 1148, 4, 1150, 4, 1152, 4, 1162, 4, 1164, 4, 1166, 4, 1168, 4, 1170, 4, 1172, 4, 1174, 4, 1176, 4, 1178, 4, 1180, 4, 1182, 4, 1184, 4, 1186, 4, 1188, 4, 1190, 4, 1192, 4, 1194, 4, 1196, 4, 1198, 4, 1200, 4, 1202, 4, 1204, 4, 1206, 4, 1208, 4, 1210, 4, 1212, 4, 1214, 4, 1216, 60, 1217, 4, 1219, 4, 1221, 4, 1223, 4, 1225, 4, 1227, 4, 1229, 4, 1232, 4, 1234, 4, 1236, 4, 1238, 4, 1240, 4, 1242, 4, 1244, 4, 1246, 4, 1248, 4, 1250, 4, 1252, 4, 1254, 4, 1256, 4, 1258, 4, 1260, 4, 1262, 4, 1264, 4, 1266, 4, 1268, 4, 1270, 4, 1272, 4, 1274, 4, 1276, 4, 1278, 4, 1280, 4, 1282, 4, 1284, 4, 1286, 4, 1288, 4, 1290, 4, 1292, 4, 1294, 4, 1296, 4, 1298, 4, 1073743153, 192, 1366, 192, 1073746080, 29056, 4293, 29056, 7680, 4, 7682, 4, 7684, 4, 7686, 4, 7688, 4, 7690, 4, 7692, 4, 7694, 4, 7696, 4, 7698, 4, 7700, 4, 7702, 4, 7704, 4, 7706, 4, 7708, 4, 7710, 4, 7712, 4, 7714, 4, 7716, 4, 7718, 4, 7720, 4, 7722, 4, 7724, 4, 7726, 4, 7728, 4, 7730, 4, 7732, 4, 7734, 4, 7736, 4, 7738, 4, 7740, 4, 7742, 4, 7744, 4, 7746, 4, 7748, 4, 7750, 4, 7752, 4, 7754, 4, 7756, 4, 7758, 4, 7760, 4, 7762, 4, 7764, 4, 7766, 4, 7768, 4, 7770, 4, 7772, 4, 7774, 4, 7776, 4, 7778, 4, 7780, 4, 7782, 4, 7784, 4, 7786, 4, 7788, 4, 7790, 4, 7792, 4, 7794, 4, 7796, 4, 7798, 4, 7800, 4, 7802, 4, 7804, 4, 7806, 4, 7808, 4, 7810, 4, 7812, 4, 7814, 4, 7816, 4, 7818, 4, 7820, 4, 7822, 4, 7824, 4, 7826, 4, 7828, 4, 7840, 4, 7842, 4, 7844, 4, 7846, 4, 7848, 4, 7850, 4, 7852, 4, 7854, 4, 7856, 4, 7858, 4, 7860, 4, 7862, 4, 7864, 4, 7866, 4, 7868, 4, 7870, 4, 7872, 4, 7874, 4, 7876, 4, 7878, 4, 7880, 4, 7882, 4, 7884, 4, 7886, 4, 7888, 4, 7890, 4, 7892, 4, 7894, 4, 7896, 4, 7898, 4, 7900, 4, 7902, 4, 7904, 4, 7906, 4, 7908, 4, 7910, 4, 7912, 4, 7914, 4, 7916, 4, 7918, 4, 7920, 4, 7922, 4, 7924, 4, 7926, 4, 7928, 4, 1073749768, -32, 7951, -32, 1073749784, -32, 7965, -32, 1073749800, -32, 7983, -32, 1073749816, -32, 7999, -32, 1073749832, -32, 8013, -32, 8025, -32, 8027, -32, 8029, -32, 8031, -32, 1073749864, -32, 8047, -32, 1073749896, -32, 8079, -32, 1073749912, -32, 8095, -32, 1073749928, -32, 8111, -32, 1073749944, -32, 8121, -32, 1073749946, -296, 8123, -296, 8124, -36, 1073749960, -344, 8139, -344, 8140, -36, 1073749976, -32, 8153, -32, 1073749978, -400, 8155, -400, 1073749992, -32, 8169, -32, 1073749994, -448, 8171, -448, 8172, -28, 1073750008, -512, 8185, -512, 1073750010, -504, 8187, -504, 8188, -36, 8486, -30068, 8490, -33532, 8491, -33048, 8498, 112, 1073750368, 64, 8559, 64, 8579, 4, 1073751222, 104, 9423, 104, 1073753088, 192, 11310, 192, 11360, 4, 11362, -42972, 11363, -15256, 11364, -42908, 11367, 4, 11369, 4, 11371, 4, 11381, 4, 11392, 4, 11394, 4, 11396, 4, 11398, 4, 11400, 4, 11402, 4, 11404, 4, 11406, 4, 11408, 4, 11410, 4, 11412, 4, 11414, 4, 11416, 4, 11418, 4, 11420, 4, 11422, 4, 11424, 4, 11426, 4, 11428, 4, 11430, 4, 11432, 4, 11434, 4, 11436, 4, 11438, 4, 11440, 4, 11442, 4, 11444, 4, 11446, 4, 11448, 4, 11450, 4, 11452, 4, 11454, 4, 11456, 4, 11458, 4, 11460, 4, 11462, 4, 11464, 4, 11466, 4, 11468, 4, 11470, 4, 11472, 4, 11474, 4, 11476, 4, 11478, 4, 11480, 4, 11482, 4, 11484, 4, 11486, 4, 11488, 4, 11490, 4 }; // NOLINT
static const MultiCharacterSpecialCase<3> kToLowercaseMultiStrings1[] = { {0, {0}} }; // NOLINT
static const uint16_t kToLowercaseTable1Size = 2;
static const int32_t kToLowercaseTable1[4] = { 1073774369, 128, 32570, 128 }; // NOLINT
static const MultiCharacterSpecialCase<3> kToLowercaseMultiStrings2[] = { {0, {0}} }; // NOLINT
static const uint16_t kToLowercaseTable2Size = 2;
static const int32_t kToLowercaseTable2[4] = { 1073742848, 160, 1063, 160 }; // NOLINT
int ToLowercase::Convert(uchar c,
                      uchar n,
                      uchar* result,
                      bool* allow_caching_ptr) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupMapping(kToLowercaseTable0,
                                     kToLowercaseTable0Size,
                                     kToLowercaseMultiStrings0,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 1: return LookupMapping(kToLowercaseTable1,
                                     kToLowercaseTable1Size,
                                     kToLowercaseMultiStrings1,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 2: return LookupMapping(kToLowercaseTable2,
                                     kToLowercaseTable2Size,
                                     kToLowercaseMultiStrings2,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    default: return 0;
  }
}

static const MultiCharacterSpecialCase<3> kToUppercaseMultiStrings0[] = { {2, {83, 83}}, {2, {700, 78}}, {2, {74, 780}}, {3, {921, 776, 769}}, {3, {933, 776, 769}}, {2, {1333, 1362}}, {2, {72, 817}}, {2, {84, 776}}, {2, {87, 778}}, {2, {89, 778}}, {2, {65, 702}}, {2, {933, 787}}, {3, {933, 787, 768}}, {3, {933, 787, 769}}, {3, {933, 787, 834}}, {2, {7944, 921}}, {2, {7945, 921}}, {2, {7946, 921}}, {2, {7947, 921}}, {2, {7948, 921}}, {2, {7949, 921}}, {2, {7950, 921}}, {2, {7951, 921}}, {2, {7944, 921}}, {2, {7945, 921}}, {2, {7946, 921}}, {2, {7947, 921}}, {2, {7948, 921}}, {2, {7949, 921}}, {2, {7950, 921}}, {2, {7951, 921}}, {2, {7976, 921}}, {2, {7977, 921}}, {2, {7978, 921}}, {2, {7979, 921}}, {2, {7980, 921}}, {2, {7981, 921}}, {2, {7982, 921}}, {2, {7983, 921}}, {2, {7976, 921}}, {2, {7977, 921}}, {2, {7978, 921}}, {2, {7979, 921}}, {2, {7980, 921}}, {2, {7981, 921}}, {2, {7982, 921}}, {2, {7983, 921}}, {2, {8040, 921}}, {2, {8041, 921}}, {2, {8042, 921}}, {2, {8043, 921}}, {2, {8044, 921}}, {2, {8045, 921}}, {2, {8046, 921}}, {2, {8047, 921}}, {2, {8040, 921}}, {2, {8041, 921}}, {2, {8042, 921}}, {2, {8043, 921}}, {2, {8044, 921}}, {2, {8045, 921}}, {2, {8046, 921}}, {2, {8047, 921}}, {2, {8122, 921}}, {2, {913, 921}}, {2, {902, 921}}, {2, {913, 834}}, {3, {913, 834, 921}}, {2, {913, 921}}, {2, {8138, 921}}, {2, {919, 921}}, {2, {905, 921}}, {2, {919, 834}}, {3, {919, 834, 921}}, {2, {919, 921}}, {3, {921, 776, 768}}, {3, {921, 776, 769}}, {2, {921, 834}}, {3, {921, 776, 834}}, {3, {933, 776, 768}}, {3, {933, 776, 769}}, {2, {929, 787}}, {2, {933, 834}}, {3, {933, 776, 834}}, {2, {8186, 921}}, {2, {937, 921}}, {2, {911, 921}}, {2, {937, 834}}, {3, {937, 834, 921}}, {2, {937, 921}}, {0, {0}} }; // NOLINT
static const uint16_t kToUppercaseTable0Size = 621;
static const int32_t kToUppercaseTable0[1242] = { 1073741921, -128, 122, -128, 181, 2972, 223, 1, 1073742048, -128, 246, -128, 1073742072, -128, 254, -128, 255, 484, 257, -4, 259, -4, 261, -4, 263, -4, 265, -4, 267, -4, 269, -4, 271, -4, 273, -4, 275, -4, 277, -4, 279, -4, 281, -4, 283, -4, 285, -4, 287, -4, 289, -4, 291, -4, 293, -4, 295, -4, 297, -4, 299, -4, 301, -4, 303, -4, 305, -928, 307, -4, 309, -4, 311, -4, 314, -4, 316, -4, 318, -4, 320, -4, 322, -4, 324, -4, 326, -4, 328, -4, 329, 5, 331, -4, 333, -4, 335, -4, 337, -4, 339, -4, 341, -4, 343, -4, 345, -4, 347, -4, 349, -4, 351, -4, 353, -4, 355, -4, 357, -4, 359, -4, 361, -4, 363, -4, 365, -4, 367, -4, 369, -4, 371, -4, 373, -4, 375, -4, 378, -4, 380, -4, 382, -4, 383, -1200, 384, 780, 387, -4, 389, -4, 392, -4, 396, -4, 402, -4, 405, 388, 409, -4, 410, 652, 414, 520, 417, -4, 419, -4, 421, -4, 424, -4, 429, -4, 432, -4, 436, -4, 438, -4, 441, -4, 445, -4, 447, 224, 453, -4, 454, -8, 456, -4, 457, -8, 459, -4, 460, -8, 462, -4, 464, -4, 466, -4, 468, -4, 470, -4, 472, -4, 474, -4, 476, -4, 477, -316, 479, -4, 481, -4, 483, -4, 485, -4, 487, -4, 489, -4, 491, -4, 493, -4, 495, -4, 496, 9, 498, -4, 499, -8, 501, -4, 505, -4, 507, -4, 509, -4, 511, -4, 513, -4, 515, -4, 517, -4, 519, -4, 521, -4, 523, -4, 525, -4, 527, -4, 529, -4, 531, -4, 533, -4, 535, -4, 537, -4, 539, -4, 541, -4, 543, -4, 547, -4, 549, -4, 551, -4, 553, -4, 555, -4, 557, -4, 559, -4, 561, -4, 563, -4, 572, -4, 578, -4, 583, -4, 585, -4, 587, -4, 589, -4, 591, -4, 595, -840, 596, -824, 1073742422, -820, 599, -820, 601, -808, 603, -812, 608, -820, 611, -828, 616, -836, 617, -844, 619, 42972, 623, -844, 626, -852, 629, -856, 637, 42908, 640, -872, 643, -872, 648, -872, 649, -276, 1073742474, -868, 651, -868, 652, -284, 658, -876, 837, 336, 1073742715, 520, 893, 520, 912, 13, 940, -152, 1073742765, -148, 943, -148, 944, 17, 1073742769, -128, 961, -128, 962, -124, 1073742787, -128, 971, -128, 972, -256, 1073742797, -252, 974, -252, 976, -248, 977, -228, 981, -188, 982, -216, 985, -4, 987, -4, 989, -4, 991, -4, 993, -4, 995, -4, 997, -4, 999, -4, 1001, -4, 1003, -4, 1005, -4, 1007, -4, 1008, -344, 1009, -320, 1010, 28, 1013, -384, 1016, -4, 1019, -4, 1073742896, -128, 1103, -128, 1073742928, -320, 1119, -320, 1121, -4, 1123, -4, 1125, -4, 1127, -4, 1129, -4, 1131, -4, 1133, -4, 1135, -4, 1137, -4, 1139, -4, 1141, -4, 1143, -4, 1145, -4, 1147, -4, 1149, -4, 1151, -4, 1153, -4, 1163, -4, 1165, -4, 1167, -4, 1169, -4, 1171, -4, 1173, -4, 1175, -4, 1177, -4, 1179, -4, 1181, -4, 1183, -4, 1185, -4, 1187, -4, 1189, -4, 1191, -4, 1193, -4, 1195, -4, 1197, -4, 1199, -4, 1201, -4, 1203, -4, 1205, -4, 1207, -4, 1209, -4, 1211, -4, 1213, -4, 1215, -4, 1218, -4, 1220, -4, 1222, -4, 1224, -4, 1226, -4, 1228, -4, 1230, -4, 1231, -60, 1233, -4, 1235, -4, 1237, -4, 1239, -4, 1241, -4, 1243, -4, 1245, -4, 1247, -4, 1249, -4, 1251, -4, 1253, -4, 1255, -4, 1257, -4, 1259, -4, 1261, -4, 1263, -4, 1265, -4, 1267, -4, 1269, -4, 1271, -4, 1273, -4, 1275, -4, 1277, -4, 1279, -4, 1281, -4, 1283, -4, 1285, -4, 1287, -4, 1289, -4, 1291, -4, 1293, -4, 1295, -4, 1297, -4, 1299, -4, 1073743201, -192, 1414, -192, 1415, 21, 7549, 15256, 7681, -4, 7683, -4, 7685, -4, 7687, -4, 7689, -4, 7691, -4, 7693, -4, 7695, -4, 7697, -4, 7699, -4, 7701, -4, 7703, -4, 7705, -4, 7707, -4, 7709, -4, 7711, -4, 7713, -4, 7715, -4, 7717, -4, 7719, -4, 7721, -4, 7723, -4, 7725, -4, 7727, -4, 7729, -4, 7731, -4, 7733, -4, 7735, -4, 7737, -4, 7739, -4, 7741, -4, 7743, -4, 7745, -4, 7747, -4, 7749, -4, 7751, -4, 7753, -4, 7755, -4, 7757, -4, 7759, -4, 7761, -4, 7763, -4, 7765, -4, 7767, -4, 7769, -4, 7771, -4, 7773, -4, 7775, -4, 7777, -4, 7779, -4, 7781, -4, 7783, -4, 7785, -4, 7787, -4, 7789, -4, 7791, -4, 7793, -4, 7795, -4, 7797, -4, 7799, -4, 7801, -4, 7803, -4, 7805, -4, 7807, -4, 7809, -4, 7811, -4, 7813, -4, 7815, -4, 7817, -4, 7819, -4, 7821, -4, 7823, -4, 7825, -4, 7827, -4, 7829, -4, 7830, 25, 7831, 29, 7832, 33, 7833, 37, 7834, 41, 7835, -236, 7841, -4, 7843, -4, 7845, -4, 7847, -4, 7849, -4, 7851, -4, 7853, -4, 7855, -4, 7857, -4, 7859, -4, 7861, -4, 7863, -4, 7865, -4, 7867, -4, 7869, -4, 7871, -4, 7873, -4, 7875, -4, 7877, -4, 7879, -4, 7881, -4, 7883, -4, 7885, -4, 7887, -4, 7889, -4, 7891, -4, 7893, -4, 7895, -4, 7897, -4, 7899, -4, 7901, -4, 7903, -4, 7905, -4, 7907, -4, 7909, -4, 7911, -4, 7913, -4, 7915, -4, 7917, -4, 7919, -4, 7921, -4, 7923, -4, 7925, -4, 7927, -4, 7929, -4, 1073749760, 32, 7943, 32, 1073749776, 32, 7957, 32, 1073749792, 32, 7975, 32, 1073749808, 32, 7991, 32, 1073749824, 32, 8005, 32, 8016, 45, 8017, 32, 8018, 49, 8019, 32, 8020, 53, 8021, 32, 8022, 57, 8023, 32, 1073749856, 32, 8039, 32, 1073749872, 296, 8049, 296, 1073749874, 344, 8053, 344, 1073749878, 400, 8055, 400, 1073749880, 512, 8057, 512, 1073749882, 448, 8059, 448, 1073749884, 504, 8061, 504, 8064, 61, 8065, 65, 8066, 69, 8067, 73, 8068, 77, 8069, 81, 8070, 85, 8071, 89, 8072, 93, 8073, 97, 8074, 101, 8075, 105, 8076, 109, 8077, 113, 8078, 117, 8079, 121, 8080, 125, 8081, 129, 8082, 133, 8083, 137, 8084, 141, 8085, 145, 8086, 149, 8087, 153, 8088, 157, 8089, 161, 8090, 165, 8091, 169, 8092, 173, 8093, 177, 8094, 181, 8095, 185, 8096, 189, 8097, 193, 8098, 197, 8099, 201, 8100, 205, 8101, 209, 8102, 213, 8103, 217, 8104, 221, 8105, 225, 8106, 229, 8107, 233, 8108, 237, 8109, 241, 8110, 245, 8111, 249, 1073749936, 32, 8113, 32, 8114, 253, 8115, 257, 8116, 261, 8118, 265, 8119, 269, 8124, 273, 8126, -28820, 8130, 277, 8131, 281, 8132, 285, 8134, 289, 8135, 293, 8140, 297, 1073749968, 32, 8145, 32, 8146, 301, 8147, 305, 8150, 309, 8151, 313, 1073749984, 32, 8161, 32, 8162, 317, 8163, 321, 8164, 325, 8165, 28, 8166, 329, 8167, 333, 8178, 337, 8179, 341, 8180, 345, 8182, 349, 8183, 353, 8188, 357, 8526, -112, 1073750384, -64, 8575, -64, 8580, -4, 1073751248, -104, 9449, -104, 1073753136, -192, 11358, -192, 11361, -4, 11365, -43180, 11366, -43168, 11368, -4, 11370, -4, 11372, -4, 11382, -4, 11393, -4, 11395, -4, 11397, -4, 11399, -4, 11401, -4, 11403, -4, 11405, -4, 11407, -4, 11409, -4, 11411, -4, 11413, -4, 11415, -4, 11417, -4, 11419, -4, 11421, -4, 11423, -4, 11425, -4, 11427, -4, 11429, -4, 11431, -4, 11433, -4, 11435, -4, 11437, -4, 11439, -4, 11441, -4, 11443, -4, 11445, -4, 11447, -4, 11449, -4, 11451, -4, 11453, -4, 11455, -4, 11457, -4, 11459, -4, 11461, -4, 11463, -4, 11465, -4, 11467, -4, 11469, -4, 11471, -4, 11473, -4, 11475, -4, 11477, -4, 11479, -4, 11481, -4, 11483, -4, 11485, -4, 11487, -4, 11489, -4, 11491, -4, 1073753344, -29056, 11557, -29056 }; // NOLINT
static const MultiCharacterSpecialCase<3> kToUppercaseMultiStrings1[] = { {2, {70, 70}}, {2, {70, 73}}, {2, {70, 76}}, {3, {70, 70, 73}}, {3, {70, 70, 76}}, {2, {83, 84}}, {2, {83, 84}}, {2, {1348, 1350}}, {2, {1348, 1333}}, {2, {1348, 1339}}, {2, {1358, 1350}}, {2, {1348, 1341}}, {0, {0}} }; // NOLINT
static const uint16_t kToUppercaseTable1Size = 14;
static const int32_t kToUppercaseTable1[28] = { 31488, 1, 31489, 5, 31490, 9, 31491, 13, 31492, 17, 31493, 21, 31494, 25, 31507, 29, 31508, 33, 31509, 37, 31510, 41, 31511, 45, 1073774401, -128, 32602, -128 }; // NOLINT
static const MultiCharacterSpecialCase<3> kToUppercaseMultiStrings2[] = { {0, {0}} }; // NOLINT
static const uint16_t kToUppercaseTable2Size = 2;
static const int32_t kToUppercaseTable2[4] = { 1073742888, -160, 1103, -160 }; // NOLINT
int ToUppercase::Convert(uchar c,
                      uchar n,
                      uchar* result,
                      bool* allow_caching_ptr) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupMapping(kToUppercaseTable0,
                                     kToUppercaseTable0Size,
                                     kToUppercaseMultiStrings0,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 1: return LookupMapping(kToUppercaseTable1,
                                     kToUppercaseTable1Size,
                                     kToUppercaseMultiStrings1,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 2: return LookupMapping(kToUppercaseTable2,
                                     kToUppercaseTable2Size,
                                     kToUppercaseMultiStrings2,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    default: return 0;
  }
}

static const MultiCharacterSpecialCase<1> kEcma262CanonicalizeMultiStrings0[] = { {0, {0}} }; // NOLINT
static const uint16_t kEcma262CanonicalizeTable0Size = 529;
static const int32_t kEcma262CanonicalizeTable0[1058] = { 1073741921, -128, 122, -128, 181, 2972, 1073742048, -128, 246, -128, 1073742072, -128, 254, -128, 255, 484, 257, -4, 259, -4, 261, -4, 263, -4, 265, -4, 267, -4, 269, -4, 271, -4, 273, -4, 275, -4, 277, -4, 279, -4, 281, -4, 283, -4, 285, -4, 287, -4, 289, -4, 291, -4, 293, -4, 295, -4, 297, -4, 299, -4, 301, -4, 303, -4, 307, -4, 309, -4, 311, -4, 314, -4, 316, -4, 318, -4, 320, -4, 322, -4, 324, -4, 326, -4, 328, -4, 331, -4, 333, -4, 335, -4, 337, -4, 339, -4, 341, -4, 343, -4, 345, -4, 347, -4, 349, -4, 351, -4, 353, -4, 355, -4, 357, -4, 359, -4, 361, -4, 363, -4, 365, -4, 367, -4, 369, -4, 371, -4, 373, -4, 375, -4, 378, -4, 380, -4, 382, -4, 384, 780, 387, -4, 389, -4, 392, -4, 396, -4, 402, -4, 405, 388, 409, -4, 410, 652, 414, 520, 417, -4, 419, -4, 421, -4, 424, -4, 429, -4, 432, -4, 436, -4, 438, -4, 441, -4, 445, -4, 447, 224, 453, -4, 454, -8, 456, -4, 457, -8, 459, -4, 460, -8, 462, -4, 464, -4, 466, -4, 468, -4, 470, -4, 472, -4, 474, -4, 476, -4, 477, -316, 479, -4, 481, -4, 483, -4, 485, -4, 487, -4, 489, -4, 491, -4, 493, -4, 495, -4, 498, -4, 499, -8, 501, -4, 505, -4, 507, -4, 509, -4, 511, -4, 513, -4, 515, -4, 517, -4, 519, -4, 521, -4, 523, -4, 525, -4, 527, -4, 529, -4, 531, -4, 533, -4, 535, -4, 537, -4, 539, -4, 541, -4, 543, -4, 547, -4, 549, -4, 551, -4, 553, -4, 555, -4, 557, -4, 559, -4, 561, -4, 563, -4, 572, -4, 578, -4, 583, -4, 585, -4, 587, -4, 589, -4, 591, -4, 595, -840, 596, -824, 1073742422, -820, 599, -820, 601, -808, 603, -812, 608, -820, 611, -828, 616, -836, 617, -844, 619, 42972, 623, -844, 626, -852, 629, -856, 637, 42908, 640, -872, 643, -872, 648, -872, 649, -276, 1073742474, -868, 651, -868, 652, -284, 658, -876, 837, 336, 1073742715, 520, 893, 520, 940, -152, 1073742765, -148, 943, -148, 1073742769, -128, 961, -128, 962, -124, 1073742787, -128, 971, -128, 972, -256, 1073742797, -252, 974, -252, 976, -248, 977, -228, 981, -188, 982, -216, 985, -4, 987, -4, 989, -4, 991, -4, 993, -4, 995, -4, 997, -4, 999, -4, 1001, -4, 1003, -4, 1005, -4, 1007, -4, 1008, -344, 1009, -320, 1010, 28, 1013, -384, 1016, -4, 1019, -4, 1073742896, -128, 1103, -128, 1073742928, -320, 1119, -320, 1121, -4, 1123, -4, 1125, -4, 1127, -4, 1129, -4, 1131, -4, 1133, -4, 1135, -4, 1137, -4, 1139, -4, 1141, -4, 1143, -4, 1145, -4, 1147, -4, 1149, -4, 1151, -4, 1153, -4, 1163, -4, 1165, -4, 1167, -4, 1169, -4, 1171, -4, 1173, -4, 1175, -4, 1177, -4, 1179, -4, 1181, -4, 1183, -4, 1185, -4, 1187, -4, 1189, -4, 1191, -4, 1193, -4, 1195, -4, 1197, -4, 1199, -4, 1201, -4, 1203, -4, 1205, -4, 1207, -4, 1209, -4, 1211, -4, 1213, -4, 1215, -4, 1218, -4, 1220, -4, 1222, -4, 1224, -4, 1226, -4, 1228, -4, 1230, -4, 1231, -60, 1233, -4, 1235, -4, 1237, -4, 1239, -4, 1241, -4, 1243, -4, 1245, -4, 1247, -4, 1249, -4, 1251, -4, 1253, -4, 1255, -4, 1257, -4, 1259, -4, 1261, -4, 1263, -4, 1265, -4, 1267, -4, 1269, -4, 1271, -4, 1273, -4, 1275, -4, 1277, -4, 1279, -4, 1281, -4, 1283, -4, 1285, -4, 1287, -4, 1289, -4, 1291, -4, 1293, -4, 1295, -4, 1297, -4, 1299, -4, 1073743201, -192, 1414, -192, 7549, 15256, 7681, -4, 7683, -4, 7685, -4, 7687, -4, 7689, -4, 7691, -4, 7693, -4, 7695, -4, 7697, -4, 7699, -4, 7701, -4, 7703, -4, 7705, -4, 7707, -4, 7709, -4, 7711, -4, 7713, -4, 7715, -4, 7717, -4, 7719, -4, 7721, -4, 7723, -4, 7725, -4, 7727, -4, 7729, -4, 7731, -4, 7733, -4, 7735, -4, 7737, -4, 7739, -4, 7741, -4, 7743, -4, 7745, -4, 7747, -4, 7749, -4, 7751, -4, 7753, -4, 7755, -4, 7757, -4, 7759, -4, 7761, -4, 7763, -4, 7765, -4, 7767, -4, 7769, -4, 7771, -4, 7773, -4, 7775, -4, 7777, -4, 7779, -4, 7781, -4, 7783, -4, 7785, -4, 7787, -4, 7789, -4, 7791, -4, 7793, -4, 7795, -4, 7797, -4, 7799, -4, 7801, -4, 7803, -4, 7805, -4, 7807, -4, 7809, -4, 7811, -4, 7813, -4, 7815, -4, 7817, -4, 7819, -4, 7821, -4, 7823, -4, 7825, -4, 7827, -4, 7829, -4, 7835, -236, 7841, -4, 7843, -4, 7845, -4, 7847, -4, 7849, -4, 7851, -4, 7853, -4, 7855, -4, 7857, -4, 7859, -4, 7861, -4, 7863, -4, 7865, -4, 7867, -4, 7869, -4, 7871, -4, 7873, -4, 7875, -4, 7877, -4, 7879, -4, 7881, -4, 7883, -4, 7885, -4, 7887, -4, 7889, -4, 7891, -4, 7893, -4, 7895, -4, 7897, -4, 7899, -4, 7901, -4, 7903, -4, 7905, -4, 7907, -4, 7909, -4, 7911, -4, 7913, -4, 7915, -4, 7917, -4, 7919, -4, 7921, -4, 7923, -4, 7925, -4, 7927, -4, 7929, -4, 1073749760, 32, 7943, 32, 1073749776, 32, 7957, 32, 1073749792, 32, 7975, 32, 1073749808, 32, 7991, 32, 1073749824, 32, 8005, 32, 8017, 32, 8019, 32, 8021, 32, 8023, 32, 1073749856, 32, 8039, 32, 1073749872, 296, 8049, 296, 1073749874, 344, 8053, 344, 1073749878, 400, 8055, 400, 1073749880, 512, 8057, 512, 1073749882, 448, 8059, 448, 1073749884, 504, 8061, 504, 1073749936, 32, 8113, 32, 8126, -28820, 1073749968, 32, 8145, 32, 1073749984, 32, 8161, 32, 8165, 28, 8526, -112, 1073750384, -64, 8575, -64, 8580, -4, 1073751248, -104, 9449, -104, 1073753136, -192, 11358, -192, 11361, -4, 11365, -43180, 11366, -43168, 11368, -4, 11370, -4, 11372, -4, 11382, -4, 11393, -4, 11395, -4, 11397, -4, 11399, -4, 11401, -4, 11403, -4, 11405, -4, 11407, -4, 11409, -4, 11411, -4, 11413, -4, 11415, -4, 11417, -4, 11419, -4, 11421, -4, 11423, -4, 11425, -4, 11427, -4, 11429, -4, 11431, -4, 11433, -4, 11435, -4, 11437, -4, 11439, -4, 11441, -4, 11443, -4, 11445, -4, 11447, -4, 11449, -4, 11451, -4, 11453, -4, 11455, -4, 11457, -4, 11459, -4, 11461, -4, 11463, -4, 11465, -4, 11467, -4, 11469, -4, 11471, -4, 11473, -4, 11475, -4, 11477, -4, 11479, -4, 11481, -4, 11483, -4, 11485, -4, 11487, -4, 11489, -4, 11491, -4, 1073753344, -29056, 11557, -29056 }; // NOLINT
static const MultiCharacterSpecialCase<1> kEcma262CanonicalizeMultiStrings1[] = { {0, {0}} }; // NOLINT
static const uint16_t kEcma262CanonicalizeTable1Size = 2;
static const int32_t kEcma262CanonicalizeTable1[4] = { 1073774401, -128, 32602, -128 }; // NOLINT
static const MultiCharacterSpecialCase<1> kEcma262CanonicalizeMultiStrings2[] = { {0, {0}} }; // NOLINT
static const uint16_t kEcma262CanonicalizeTable2Size = 2;
static const int32_t kEcma262CanonicalizeTable2[4] = { 1073742888, -160, 1103, -160 }; // NOLINT
int Ecma262Canonicalize::Convert(uchar c,
                      uchar n,
                      uchar* result,
                      bool* allow_caching_ptr) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupMapping(kEcma262CanonicalizeTable0,
                                     kEcma262CanonicalizeTable0Size,
                                     kEcma262CanonicalizeMultiStrings0,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 1: return LookupMapping(kEcma262CanonicalizeTable1,
                                     kEcma262CanonicalizeTable1Size,
                                     kEcma262CanonicalizeMultiStrings1,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 2: return LookupMapping(kEcma262CanonicalizeTable2,
                                     kEcma262CanonicalizeTable2Size,
                                     kEcma262CanonicalizeMultiStrings2,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    default: return 0;
  }
}

static const MultiCharacterSpecialCase<4> kEcma262UnCanonicalizeMultiStrings0[] = { {2, {65, 97}}, {2, {66, 98}}, {2, {67, 99}}, {2, {68, 100}}, {2, {69, 101}}, {2, {70, 102}}, {2, {71, 103}}, {2, {72, 104}}, {2, {73, 105}}, {2, {74, 106}}, {2, {75, 107}}, {2, {76, 108}}, {2, {77, 109}}, {2, {78, 110}}, {2, {79, 111}}, {2, {80, 112}}, {2, {81, 113}}, {2, {82, 114}}, {2, {83, 115}}, {2, {84, 116}}, {2, {85, 117}}, {2, {86, 118}}, {2, {87, 119}}, {2, {88, 120}}, {2, {89, 121}}, {2, {90, 122}}, {2, {65, 97}}, {2, {66, 98}}, {2, {67, 99}}, {2, {68, 100}}, {2, {69, 101}}, {2, {70, 102}}, {2, {71, 103}}, {2, {72, 104}}, {2, {73, 105}}, {2, {74, 106}}, {2, {75, 107}}, {2, {76, 108}}, {2, {77, 109}}, {2, {78, 110}}, {2, {79, 111}}, {2, {80, 112}}, {2, {81, 113}}, {2, {82, 114}}, {2, {83, 115}}, {2, {84, 116}}, {2, {85, 117}}, {2, {86, 118}}, {2, {87, 119}}, {2, {88, 120}}, {2, {89, 121}}, {2, {90, 122}}, {3, {181, 924, 956}}, {2, {192, 224}}, {2, {193, 225}}, {2, {194, 226}}, {2, {195, 227}}, {2, {196, 228}}, {2, {197, 229}}, {2, {198, 230}}, {2, {199, 231}}, {2, {200, 232}}, {2, {201, 233}}, {2, {202, 234}}, {2, {203, 235}}, {2, {204, 236}}, {2, {205, 237}}, {2, {206, 238}}, {2, {207, 239}}, {2, {208, 240}}, {2, {209, 241}}, {2, {210, 242}}, {2, {211, 243}}, {2, {212, 244}}, {2, {213, 245}}, {2, {214, 246}}, {2, {216, 248}}, {2, {217, 249}}, {2, {218, 250}}, {2, {219, 251}}, {2, {220, 252}}, {2, {221, 253}}, {2, {222, 254}}, {2, {192, 224}}, {2, {193, 225}}, {2, {194, 226}}, {2, {195, 227}}, {2, {196, 228}}, {2, {197, 229}}, {2, {198, 230}}, {2, {199, 231}}, {2, {200, 232}}, {2, {201, 233}}, {2, {202, 234}}, {2, {203, 235}}, {2, {204, 236}}, {2, {205, 237}}, {2, {206, 238}}, {2, {207, 239}}, {2, {208, 240}}, {2, {209, 241}}, {2, {210, 242}}, {2, {211, 243}}, {2, {212, 244}}, {2, {213, 245}}, {2, {214, 246}}, {2, {216, 248}}, {2, {217, 249}}, {2, {218, 250}}, {2, {219, 251}}, {2, {220, 252}}, {2, {221, 253}}, {2, {222, 254}}, {2, {255, 376}}, {2, {256, 257}}, {2, {256, 257}}, {2, {258, 259}}, {2, {258, 259}}, {2, {260, 261}}, {2, {260, 261}}, {2, {262, 263}}, {2, {262, 263}}, {2, {264, 265}}, {2, {264, 265}}, {2, {266, 267}}, {2, {266, 267}}, {2, {268, 269}}, {2, {268, 269}}, {2, {270, 271}}, {2, {270, 271}}, {2, {272, 273}}, {2, {272, 273}}, {2, {274, 275}}, {2, {274, 275}}, {2, {276, 277}}, {2, {276, 277}}, {2, {278, 279}}, {2, {278, 279}}, {2, {280, 281}}, {2, {280, 281}}, {2, {282, 283}}, {2, {282, 283}}, {2, {284, 285}}, {2, {284, 285}}, {2, {286, 287}}, {2, {286, 287}}, {2, {288, 289}}, {2, {288, 289}}, {2, {290, 291}}, {2, {290, 291}}, {2, {292, 293}}, {2, {292, 293}}, {2, {294, 295}}, {2, {294, 295}}, {2, {296, 297}}, {2, {296, 297}}, {2, {298, 299}}, {2, {298, 299}}, {2, {300, 301}}, {2, {300, 301}}, {2, {302, 303}}, {2, {302, 303}}, {2, {306, 307}}, {2, {306, 307}}, {2, {308, 309}}, {2, {308, 309}}, {2, {310, 311}}, {2, {310, 311}}, {2, {313, 314}}, {2, {313, 314}}, {2, {315, 316}}, {2, {315, 316}}, {2, {317, 318}}, {2, {317, 318}}, {2, {319, 320}}, {2, {319, 320}}, {2, {321, 322}}, {2, {321, 322}}, {2, {323, 324}}, {2, {323, 324}}, {2, {325, 326}}, {2, {325, 326}}, {2, {327, 328}}, {2, {327, 328}}, {2, {330, 331}}, {2, {330, 331}}, {2, {332, 333}}, {2, {332, 333}}, {2, {334, 335}}, {2, {334, 335}}, {2, {336, 337}}, {2, {336, 337}}, {2, {338, 339}}, {2, {338, 339}}, {2, {340, 341}}, {2, {340, 341}}, {2, {342, 343}}, {2, {342, 343}}, {2, {344, 345}}, {2, {344, 345}}, {2, {346, 347}}, {2, {346, 347}}, {2, {348, 349}}, {2, {348, 349}}, {2, {350, 351}}, {2, {350, 351}}, {2, {352, 353}}, {2, {352, 353}}, {2, {354, 355}}, {2, {354, 355}}, {2, {356, 357}}, {2, {356, 357}}, {2, {358, 359}}, {2, {358, 359}}, {2, {360, 361}}, {2, {360, 361}}, {2, {362, 363}}, {2, {362, 363}}, {2, {364, 365}}, {2, {364, 365}}, {2, {366, 367}}, {2, {366, 367}}, {2, {368, 369}}, {2, {368, 369}}, {2, {370, 371}}, {2, {370, 371}}, {2, {372, 373}}, {2, {372, 373}}, {2, {374, 375}}, {2, {374, 375}}, {2, {255, 376}}, {2, {377, 378}}, {2, {377, 378}}, {2, {379, 380}}, {2, {379, 380}}, {2, {381, 382}}, {2, {381, 382}}, {2, {384, 579}}, {2, {385, 595}}, {2, {386, 387}}, {2, {386, 387}}, {2, {388, 389}}, {2, {388, 389}}, {2, {390, 596}}, {2, {391, 392}}, {2, {391, 392}}, {2, {393, 598}}, {2, {394, 599}}, {2, {395, 396}}, {2, {395, 396}}, {2, {398, 477}}, {2, {399, 601}}, {2, {400, 603}}, {2, {401, 402}}, {2, {401, 402}}, {2, {403, 608}}, {2, {404, 611}}, {2, {405, 502}}, {2, {406, 617}}, {2, {407, 616}}, {2, {408, 409}}, {2, {408, 409}}, {2, {410, 573}}, {2, {412, 623}}, {2, {413, 626}}, {2, {414, 544}}, {2, {415, 629}}, {2, {416, 417}}, {2, {416, 417}}, {2, {418, 419}}, {2, {418, 419}}, {2, {420, 421}}, {2, {420, 421}}, {2, {422, 640}}, {2, {423, 424}}, {2, {423, 424}}, {2, {425, 643}}, {2, {428, 429}}, {2, {428, 429}}, {2, {430, 648}}, {2, {431, 432}}, {2, {431, 432}}, {2, {433, 650}}, {2, {434, 651}}, {2, {435, 436}}, {2, {435, 436}}, {2, {437, 438}}, {2, {437, 438}}, {2, {439, 658}}, {2, {440, 441}}, {2, {440, 441}}, {2, {444, 445}}, {2, {444, 445}}, {2, {447, 503}}, {3, {452, 453, 454}}, {3, {452, 453, 454}}, {3, {452, 453, 454}}, {3, {455, 456, 457}}, {3, {455, 456, 457}}, {3, {455, 456, 457}}, {3, {458, 459, 460}}, {3, {458, 459, 460}}, {3, {458, 459, 460}}, {2, {461, 462}}, {2, {461, 462}}, {2, {463, 464}}, {2, {463, 464}}, {2, {465, 466}}, {2, {465, 466}}, {2, {467, 468}}, {2, {467, 468}}, {2, {469, 470}}, {2, {469, 470}}, {2, {471, 472}}, {2, {471, 472}}, {2, {473, 474}}, {2, {473, 474}}, {2, {475, 476}}, {2, {475, 476}}, {2, {398, 477}}, {2, {478, 479}}, {2, {478, 479}}, {2, {480, 481}}, {2, {480, 481}}, {2, {482, 483}}, {2, {482, 483}}, {2, {484, 485}}, {2, {484, 485}}, {2, {486, 487}}, {2, {486, 487}}, {2, {488, 489}}, {2, {488, 489}}, {2, {490, 491}}, {2, {490, 491}}, {2, {492, 493}}, {2, {492, 493}}, {2, {494, 495}}, {2, {494, 495}}, {3, {497, 498, 499}}, {3, {497, 498, 499}}, {3, {497, 498, 499}}, {2, {500, 501}}, {2, {500, 501}}, {2, {405, 502}}, {2, {447, 503}}, {2, {504, 505}}, {2, {504, 505}}, {2, {506, 507}}, {2, {506, 507}}, {2, {508, 509}}, {2, {508, 509}}, {2, {510, 511}}, {2, {510, 511}}, {2, {512, 513}}, {2, {512, 513}}, {2, {514, 515}}, {2, {514, 515}}, {2, {516, 517}}, {2, {516, 517}}, {2, {518, 519}}, {2, {518, 519}}, {2, {520, 521}}, {2, {520, 521}}, {2, {522, 523}}, {2, {522, 523}}, {2, {524, 525}}, {2, {524, 525}}, {2, {526, 527}}, {2, {526, 527}}, {2, {528, 529}}, {2, {528, 529}}, {2, {530, 531}}, {2, {530, 531}}, {2, {532, 533}}, {2, {532, 533}}, {2, {534, 535}}, {2, {534, 535}}, {2, {536, 537}}, {2, {536, 537}}, {2, {538, 539}}, {2, {538, 539}}, {2, {540, 541}}, {2, {540, 541}}, {2, {542, 543}}, {2, {542, 543}}, {2, {414, 544}}, {2, {546, 547}}, {2, {546, 547}}, {2, {548, 549}}, {2, {548, 549}}, {2, {550, 551}}, {2, {550, 551}}, {2, {552, 553}}, {2, {552, 553}}, {2, {554, 555}}, {2, {554, 555}}, {2, {556, 557}}, {2, {556, 557}}, {2, {558, 559}}, {2, {558, 559}}, {2, {560, 561}}, {2, {560, 561}}, {2, {562, 563}}, {2, {562, 563}}, {2, {570, 11365}}, {2, {571, 572}}, {2, {571, 572}}, {2, {410, 573}}, {2, {574, 11366}}, {2, {577, 578}}, {2, {577, 578}}, {2, {384, 579}}, {2, {580, 649}}, {2, {581, 652}}, {2, {582, 583}}, {2, {582, 583}}, {2, {584, 585}}, {2, {584, 585}}, {2, {586, 587}}, {2, {586, 587}}, {2, {588, 589}}, {2, {588, 589}}, {2, {590, 591}}, {2, {590, 591}}, {2, {385, 595}}, {2, {390, 596}}, {2, {393, 598}}, {2, {394, 599}}, {2, {399, 601}}, {2, {400, 603}}, {2, {403, 608}}, {2, {404, 611}}, {2, {407, 616}}, {2, {406, 617}}, {2, {619, 11362}}, {2, {412, 623}}, {2, {413, 626}}, {2, {415, 629}}, {2, {637, 11364}}, {2, {422, 640}}, {2, {425, 643}}, {2, {430, 648}}, {2, {580, 649}}, {2, {433, 650}}, {2, {434, 651}}, {2, {581, 652}}, {2, {439, 658}}, {4, {837, 921, 953, 8126}}, {2, {891, 1021}}, {2, {892, 1022}}, {2, {893, 1023}}, {2, {902, 940}}, {2, {904, 941}}, {2, {905, 942}}, {2, {906, 943}}, {2, {908, 972}}, {2, {910, 973}}, {2, {911, 974}}, {2, {913, 945}}, {3, {914, 946, 976}}, {2, {915, 947}}, {2, {916, 948}}, {3, {917, 949, 1013}}, {2, {918, 950}}, {2, {919, 951}}, {3, {920, 952, 977}}, {4, {837, 921, 953, 8126}}, {3, {922, 954, 1008}}, {2, {923, 955}}, {3, {181, 924, 956}}, {2, {925, 957}}, {2, {926, 958}}, {2, {927, 959}}, {3, {928, 960, 982}}, {3, {929, 961, 1009}}, {3, {931, 962, 963}}, {2, {932, 964}}, {2, {933, 965}}, {3, {934, 966, 981}}, {2, {935, 967}}, {2, {936, 968}}, {2, {937, 969}}, {2, {938, 970}}, {2, {939, 971}}, {2, {902, 940}}, {2, {904, 941}}, {2, {905, 942}}, {2, {906, 943}}, {2, {913, 945}}, {3, {914, 946, 976}}, {2, {915, 947}}, {2, {916, 948}}, {3, {917, 949, 1013}}, {2, {918, 950}}, {2, {919, 951}}, {3, {920, 952, 977}}, {4, {837, 921, 953, 8126}}, {3, {922, 954, 1008}}, {2, {923, 955}}, {3, {181, 924, 956}}, {2, {925, 957}}, {2, {926, 958}}, {2, {927, 959}}, {3, {928, 960, 982}}, {3, {929, 961, 1009}}, {3, {931, 962, 963}}, {3, {931, 962, 963}}, {2, {932, 964}}, {2, {933, 965}}, {3, {934, 966, 981}}, {2, {935, 967}}, {2, {936, 968}}, {2, {937, 969}}, {2, {938, 970}}, {2, {939, 971}}, {2, {908, 972}}, {2, {910, 973}}, {2, {911, 974}}, {3, {914, 946, 976}}, {3, {920, 952, 977}}, {3, {934, 966, 981}}, {3, {928, 960, 982}}, {2, {984, 985}}, {2, {984, 985}}, {2, {986, 987}}, {2, {986, 987}}, {2, {988, 989}}, {2, {988, 989}}, {2, {990, 991}}, {2, {990, 991}}, {2, {992, 993}}, {2, {992, 993}}, {2, {994, 995}}, {2, {994, 995}}, {2, {996, 997}}, {2, {996, 997}}, {2, {998, 999}}, {2, {998, 999}}, {2, {1000, 1001}}, {2, {1000, 1001}}, {2, {1002, 1003}}, {2, {1002, 1003}}, {2, {1004, 1005}}, {2, {1004, 1005}}, {2, {1006, 1007}}, {2, {1006, 1007}}, {3, {922, 954, 1008}}, {3, {929, 961, 1009}}, {2, {1010, 1017}}, {3, {917, 949, 1013}}, {2, {1015, 1016}}, {2, {1015, 1016}}, {2, {1010, 1017}}, {2, {1018, 1019}}, {2, {1018, 1019}}, {2, {891, 1021}}, {2, {892, 1022}}, {2, {893, 1023}}, {2, {1024, 1104}}, {2, {1025, 1105}}, {2, {1026, 1106}}, {2, {1027, 1107}}, {2, {1028, 1108}}, {2, {1029, 1109}}, {2, {1030, 1110}}, {2, {1031, 1111}}, {2, {1032, 1112}}, {2, {1033, 1113}}, {2, {1034, 1114}}, {2, {1035, 1115}}, {2, {1036, 1116}}, {2, {1037, 1117}}, {2, {1038, 1118}}, {2, {1039, 1119}}, {2, {1040, 1072}}, {2, {1041, 1073}}, {2, {1042, 1074}}, {2, {1043, 1075}}, {2, {1044, 1076}}, {2, {1045, 1077}}, {2, {1046, 1078}}, {2, {1047, 1079}}, {2, {1048, 1080}}, {2, {1049, 1081}}, {2, {1050, 1082}}, {2, {1051, 1083}}, {2, {1052, 1084}}, {2, {1053, 1085}}, {2, {1054, 1086}}, {2, {1055, 1087}}, {2, {1056, 1088}}, {2, {1057, 1089}}, {2, {1058, 1090}}, {2, {1059, 1091}}, {2, {1060, 1092}}, {2, {1061, 1093}}, {2, {1062, 1094}}, {2, {1063, 1095}}, {2, {1064, 1096}}, {2, {1065, 1097}}, {2, {1066, 1098}}, {2, {1067, 1099}}, {2, {1068, 1100}}, {2, {1069, 1101}}, {2, {1070, 1102}}, {2, {1071, 1103}}, {2, {1040, 1072}}, {2, {1041, 1073}}, {2, {1042, 1074}}, {2, {1043, 1075}}, {2, {1044, 1076}}, {2, {1045, 1077}}, {2, {1046, 1078}}, {2, {1047, 1079}}, {2, {1048, 1080}}, {2, {1049, 1081}}, {2, {1050, 1082}}, {2, {1051, 1083}}, {2, {1052, 1084}}, {2, {1053, 1085}}, {2, {1054, 1086}}, {2, {1055, 1087}}, {2, {1056, 1088}}, {2, {1057, 1089}}, {2, {1058, 1090}}, {2, {1059, 1091}}, {2, {1060, 1092}}, {2, {1061, 1093}}, {2, {1062, 1094}}, {2, {1063, 1095}}, {2, {1064, 1096}}, {2, {1065, 1097}}, {2, {1066, 1098}}, {2, {1067, 1099}}, {2, {1068, 1100}}, {2, {1069, 1101}}, {2, {1070, 1102}}, {2, {1071, 1103}}, {2, {1024, 1104}}, {2, {1025, 1105}}, {2, {1026, 1106}}, {2, {1027, 1107}}, {2, {1028, 1108}}, {2, {1029, 1109}}, {2, {1030, 1110}}, {2, {1031, 1111}}, {2, {1032, 1112}}, {2, {1033, 1113}}, {2, {1034, 1114}}, {2, {1035, 1115}}, {2, {1036, 1116}}, {2, {1037, 1117}}, {2, {1038, 1118}}, {2, {1039, 1119}}, {2, {1120, 1121}}, {2, {1120, 1121}}, {2, {1122, 1123}}, {2, {1122, 1123}}, {2, {1124, 1125}}, {2, {1124, 1125}}, {2, {1126, 1127}}, {2, {1126, 1127}}, {2, {1128, 1129}}, {2, {1128, 1129}}, {2, {1130, 1131}}, {2, {1130, 1131}}, {2, {1132, 1133}}, {2, {1132, 1133}}, {2, {1134, 1135}}, {2, {1134, 1135}}, {2, {1136, 1137}}, {2, {1136, 1137}}, {2, {1138, 1139}}, {2, {1138, 1139}}, {2, {1140, 1141}}, {2, {1140, 1141}}, {2, {1142, 1143}}, {2, {1142, 1143}}, {2, {1144, 1145}}, {2, {1144, 1145}}, {2, {1146, 1147}}, {2, {1146, 1147}}, {2, {1148, 1149}}, {2, {1148, 1149}}, {2, {1150, 1151}}, {2, {1150, 1151}}, {2, {1152, 1153}}, {2, {1152, 1153}}, {2, {1162, 1163}}, {2, {1162, 1163}}, {2, {1164, 1165}}, {2, {1164, 1165}}, {2, {1166, 1167}}, {2, {1166, 1167}}, {2, {1168, 1169}}, {2, {1168, 1169}}, {2, {1170, 1171}}, {2, {1170, 1171}}, {2, {1172, 1173}}, {2, {1172, 1173}}, {2, {1174, 1175}}, {2, {1174, 1175}}, {2, {1176, 1177}}, {2, {1176, 1177}}, {2, {1178, 1179}}, {2, {1178, 1179}}, {2, {1180, 1181}}, {2, {1180, 1181}}, {2, {1182, 1183}}, {2, {1182, 1183}}, {2, {1184, 1185}}, {2, {1184, 1185}}, {2, {1186, 1187}}, {2, {1186, 1187}}, {2, {1188, 1189}}, {2, {1188, 1189}}, {2, {1190, 1191}}, {2, {1190, 1191}}, {2, {1192, 1193}}, {2, {1192, 1193}}, {2, {1194, 1195}}, {2, {1194, 1195}}, {2, {1196, 1197}}, {2, {1196, 1197}}, {2, {1198, 1199}}, {2, {1198, 1199}}, {2, {1200, 1201}}, {2, {1200, 1201}}, {2, {1202, 1203}}, {2, {1202, 1203}}, {2, {1204, 1205}}, {2, {1204, 1205}}, {2, {1206, 1207}}, {2, {1206, 1207}}, {2, {1208, 1209}}, {2, {1208, 1209}}, {2, {1210, 1211}}, {2, {1210, 1211}}, {2, {1212, 1213}}, {2, {1212, 1213}}, {2, {1214, 1215}}, {2, {1214, 1215}}, {2, {1216, 1231}}, {2, {1217, 1218}}, {2, {1217, 1218}}, {2, {1219, 1220}}, {2, {1219, 1220}}, {2, {1221, 1222}}, {2, {1221, 1222}}, {2, {1223, 1224}}, {2, {1223, 1224}}, {2, {1225, 1226}}, {2, {1225, 1226}}, {2, {1227, 1228}}, {2, {1227, 1228}}, {2, {1229, 1230}}, {2, {1229, 1230}}, {2, {1216, 1231}}, {2, {1232, 1233}}, {2, {1232, 1233}}, {2, {1234, 1235}}, {2, {1234, 1235}}, {2, {1236, 1237}}, {2, {1236, 1237}}, {2, {1238, 1239}}, {2, {1238, 1239}}, {2, {1240, 1241}}, {2, {1240, 1241}}, {2, {1242, 1243}}, {2, {1242, 1243}}, {2, {1244, 1245}}, {2, {1244, 1245}}, {2, {1246, 1247}}, {2, {1246, 1247}}, {2, {1248, 1249}}, {2, {1248, 1249}}, {2, {1250, 1251}}, {2, {1250, 1251}}, {2, {1252, 1253}}, {2, {1252, 1253}}, {2, {1254, 1255}}, {2, {1254, 1255}}, {2, {1256, 1257}}, {2, {1256, 1257}}, {2, {1258, 1259}}, {2, {1258, 1259}}, {2, {1260, 1261}}, {2, {1260, 1261}}, {2, {1262, 1263}}, {2, {1262, 1263}}, {2, {1264, 1265}}, {2, {1264, 1265}}, {2, {1266, 1267}}, {2, {1266, 1267}}, {2, {1268, 1269}}, {2, {1268, 1269}}, {2, {1270, 1271}}, {2, {1270, 1271}}, {2, {1272, 1273}}, {2, {1272, 1273}}, {2, {1274, 1275}}, {2, {1274, 1275}}, {2, {1276, 1277}}, {2, {1276, 1277}}, {2, {1278, 1279}}, {2, {1278, 1279}}, {2, {1280, 1281}}, {2, {1280, 1281}}, {2, {1282, 1283}}, {2, {1282, 1283}}, {2, {1284, 1285}}, {2, {1284, 1285}}, {2, {1286, 1287}}, {2, {1286, 1287}}, {2, {1288, 1289}}, {2, {1288, 1289}}, {2, {1290, 1291}}, {2, {1290, 1291}}, {2, {1292, 1293}}, {2, {1292, 1293}}, {2, {1294, 1295}}, {2, {1294, 1295}}, {2, {1296, 1297}}, {2, {1296, 1297}}, {2, {1298, 1299}}, {2, {1298, 1299}}, {2, {1329, 1377}}, {2, {1330, 1378}}, {2, {1331, 1379}}, {2, {1332, 1380}}, {2, {1333, 1381}}, {2, {1334, 1382}}, {2, {1335, 1383}}, {2, {1336, 1384}}, {2, {1337, 1385}}, {2, {1338, 1386}}, {2, {1339, 1387}}, {2, {1340, 1388}}, {2, {1341, 1389}}, {2, {1342, 1390}}, {2, {1343, 1391}}, {2, {1344, 1392}}, {2, {1345, 1393}}, {2, {1346, 1394}}, {2, {1347, 1395}}, {2, {1348, 1396}}, {2, {1349, 1397}}, {2, {1350, 1398}}, {2, {1351, 1399}}, {2, {1352, 1400}}, {2, {1353, 1401}}, {2, {1354, 1402}}, {2, {1355, 1403}}, {2, {1356, 1404}}, {2, {1357, 1405}}, {2, {1358, 1406}}, {2, {1359, 1407}}, {2, {1360, 1408}}, {2, {1361, 1409}}, {2, {1362, 1410}}, {2, {1363, 1411}}, {2, {1364, 1412}}, {2, {1365, 1413}}, {2, {1366, 1414}}, {2, {1329, 1377}}, {2, {1330, 1378}}, {2, {1331, 1379}}, {2, {1332, 1380}}, {2, {1333, 1381}}, {2, {1334, 1382}}, {2, {1335, 1383}}, {2, {1336, 1384}}, {2, {1337, 1385}}, {2, {1338, 1386}}, {2, {1339, 1387}}, {2, {1340, 1388}}, {2, {1341, 1389}}, {2, {1342, 1390}}, {2, {1343, 1391}}, {2, {1344, 1392}}, {2, {1345, 1393}}, {2, {1346, 1394}}, {2, {1347, 1395}}, {2, {1348, 1396}}, {2, {1349, 1397}}, {2, {1350, 1398}}, {2, {1351, 1399}}, {2, {1352, 1400}}, {2, {1353, 1401}}, {2, {1354, 1402}}, {2, {1355, 1403}}, {2, {1356, 1404}}, {2, {1357, 1405}}, {2, {1358, 1406}}, {2, {1359, 1407}}, {2, {1360, 1408}}, {2, {1361, 1409}}, {2, {1362, 1410}}, {2, {1363, 1411}}, {2, {1364, 1412}}, {2, {1365, 1413}}, {2, {1366, 1414}}, {2, {4256, 11520}}, {2, {4257, 11521}}, {2, {4258, 11522}}, {2, {4259, 11523}}, {2, {4260, 11524}}, {2, {4261, 11525}}, {2, {4262, 11526}}, {2, {4263, 11527}}, {2, {4264, 11528}}, {2, {4265, 11529}}, {2, {4266, 11530}}, {2, {4267, 11531}}, {2, {4268, 11532}}, {2, {4269, 11533}}, {2, {4270, 11534}}, {2, {4271, 11535}}, {2, {4272, 11536}}, {2, {4273, 11537}}, {2, {4274, 11538}}, {2, {4275, 11539}}, {2, {4276, 11540}}, {2, {4277, 11541}}, {2, {4278, 11542}}, {2, {4279, 11543}}, {2, {4280, 11544}}, {2, {4281, 11545}}, {2, {4282, 11546}}, {2, {4283, 11547}}, {2, {4284, 11548}}, {2, {4285, 11549}}, {2, {4286, 11550}}, {2, {4287, 11551}}, {2, {4288, 11552}}, {2, {4289, 11553}}, {2, {4290, 11554}}, {2, {4291, 11555}}, {2, {4292, 11556}}, {2, {4293, 11557}}, {2, {7549, 11363}}, {2, {7680, 7681}}, {2, {7680, 7681}}, {2, {7682, 7683}}, {2, {7682, 7683}}, {2, {7684, 7685}}, {2, {7684, 7685}}, {2, {7686, 7687}}, {2, {7686, 7687}}, {2, {7688, 7689}}, {2, {7688, 7689}}, {2, {7690, 7691}}, {2, {7690, 7691}}, {2, {7692, 7693}}, {2, {7692, 7693}}, {2, {7694, 7695}}, {2, {7694, 7695}}, {2, {7696, 7697}}, {2, {7696, 7697}}, {2, {7698, 7699}}, {2, {7698, 7699}}, {2, {7700, 7701}}, {2, {7700, 7701}}, {2, {7702, 7703}}, {2, {7702, 7703}}, {2, {7704, 7705}}, {2, {7704, 7705}}, {2, {7706, 7707}}, {2, {7706, 7707}}, {2, {7708, 7709}}, {2, {7708, 7709}}, {2, {7710, 7711}}, {2, {7710, 7711}}, {2, {7712, 7713}}, {2, {7712, 7713}}, {2, {7714, 7715}}, {2, {7714, 7715}}, {2, {7716, 7717}}, {2, {7716, 7717}}, {2, {7718, 7719}}, {2, {7718, 7719}}, {2, {7720, 7721}}, {2, {7720, 7721}}, {2, {7722, 7723}}, {2, {7722, 7723}}, {2, {7724, 7725}}, {2, {7724, 7725}}, {2, {7726, 7727}}, {2, {7726, 7727}}, {2, {7728, 7729}}, {2, {7728, 7729}}, {2, {7730, 7731}}, {2, {7730, 7731}}, {2, {7732, 7733}}, {2, {7732, 7733}}, {2, {7734, 7735}}, {2, {7734, 7735}}, {2, {7736, 7737}}, {2, {7736, 7737}}, {2, {7738, 7739}}, {2, {7738, 7739}}, {2, {7740, 7741}}, {2, {7740, 7741}}, {2, {7742, 7743}}, {2, {7742, 7743}}, {2, {7744, 7745}}, {2, {7744, 7745}}, {2, {7746, 7747}}, {2, {7746, 7747}}, {2, {7748, 7749}}, {2, {7748, 7749}}, {2, {7750, 7751}}, {2, {7750, 7751}}, {2, {7752, 7753}}, {2, {7752, 7753}}, {2, {7754, 7755}}, {2, {7754, 7755}}, {2, {7756, 7757}}, {2, {7756, 7757}}, {2, {7758, 7759}}, {2, {7758, 7759}}, {2, {7760, 7761}}, {2, {7760, 7761}}, {2, {7762, 7763}}, {2, {7762, 7763}}, {2, {7764, 7765}}, {2, {7764, 7765}}, {2, {7766, 7767}}, {2, {7766, 7767}}, {2, {7768, 7769}}, {2, {7768, 7769}}, {2, {7770, 7771}}, {2, {7770, 7771}}, {2, {7772, 7773}}, {2, {7772, 7773}}, {2, {7774, 7775}}, {2, {7774, 7775}}, {3, {7776, 7777, 7835}}, {3, {7776, 7777, 7835}}, {2, {7778, 7779}}, {2, {7778, 7779}}, {2, {7780, 7781}}, {2, {7780, 7781}}, {2, {7782, 7783}}, {2, {7782, 7783}}, {2, {7784, 7785}}, {2, {7784, 7785}}, {2, {7786, 7787}}, {2, {7786, 7787}}, {2, {7788, 7789}}, {2, {7788, 7789}}, {2, {7790, 7791}}, {2, {7790, 7791}}, {2, {7792, 7793}}, {2, {7792, 7793}}, {2, {7794, 7795}}, {2, {7794, 7795}}, {2, {7796, 7797}}, {2, {7796, 7797}}, {2, {7798, 7799}}, {2, {7798, 7799}}, {2, {7800, 7801}}, {2, {7800, 7801}}, {2, {7802, 7803}}, {2, {7802, 7803}}, {2, {7804, 7805}}, {2, {7804, 7805}}, {2, {7806, 7807}}, {2, {7806, 7807}}, {2, {7808, 7809}}, {2, {7808, 7809}}, {2, {7810, 7811}}, {2, {7810, 7811}}, {2, {7812, 7813}}, {2, {7812, 7813}}, {2, {7814, 7815}}, {2, {7814, 7815}}, {2, {7816, 7817}}, {2, {7816, 7817}}, {2, {7818, 7819}}, {2, {7818, 7819}}, {2, {7820, 7821}}, {2, {7820, 7821}}, {2, {7822, 7823}}, {2, {7822, 7823}}, {2, {7824, 7825}}, {2, {7824, 7825}}, {2, {7826, 7827}}, {2, {7826, 7827}}, {2, {7828, 7829}}, {2, {7828, 7829}}, {3, {7776, 7777, 7835}}, {2, {7840, 7841}}, {2, {7840, 7841}}, {2, {7842, 7843}}, {2, {7842, 7843}}, {2, {7844, 7845}}, {2, {7844, 7845}}, {2, {7846, 7847}}, {2, {7846, 7847}}, {2, {7848, 7849}}, {2, {7848, 7849}}, {2, {7850, 7851}}, {2, {7850, 7851}}, {2, {7852, 7853}}, {2, {7852, 7853}}, {2, {7854, 7855}}, {2, {7854, 7855}}, {2, {7856, 7857}}, {2, {7856, 7857}}, {2, {7858, 7859}}, {2, {7858, 7859}}, {2, {7860, 7861}}, {2, {7860, 7861}}, {2, {7862, 7863}}, {2, {7862, 7863}}, {2, {7864, 7865}}, {2, {7864, 7865}}, {2, {7866, 7867}}, {2, {7866, 7867}}, {2, {7868, 7869}}, {2, {7868, 7869}}, {2, {7870, 7871}}, {2, {7870, 7871}}, {2, {7872, 7873}}, {2, {7872, 7873}}, {2, {7874, 7875}}, {2, {7874, 7875}}, {2, {7876, 7877}}, {2, {7876, 7877}}, {2, {7878, 7879}}, {2, {7878, 7879}}, {2, {7880, 7881}}, {2, {7880, 7881}}, {2, {7882, 7883}}, {2, {7882, 7883}}, {2, {7884, 7885}}, {2, {7884, 7885}}, {2, {7886, 7887}}, {2, {7886, 7887}}, {2, {7888, 7889}}, {2, {7888, 7889}}, {2, {7890, 7891}}, {2, {7890, 7891}}, {2, {7892, 7893}}, {2, {7892, 7893}}, {2, {7894, 7895}}, {2, {7894, 7895}}, {2, {7896, 7897}}, {2, {7896, 7897}}, {2, {7898, 7899}}, {2, {7898, 7899}}, {2, {7900, 7901}}, {2, {7900, 7901}}, {2, {7902, 7903}}, {2, {7902, 7903}}, {2, {7904, 7905}}, {2, {7904, 7905}}, {2, {7906, 7907}}, {2, {7906, 7907}}, {2, {7908, 7909}}, {2, {7908, 7909}}, {2, {7910, 7911}}, {2, {7910, 7911}}, {2, {7912, 7913}}, {2, {7912, 7913}}, {2, {7914, 7915}}, {2, {7914, 7915}}, {2, {7916, 7917}}, {2, {7916, 7917}}, {2, {7918, 7919}}, {2, {7918, 7919}}, {2, {7920, 7921}}, {2, {7920, 7921}}, {2, {7922, 7923}}, {2, {7922, 7923}}, {2, {7924, 7925}}, {2, {7924, 7925}}, {2, {7926, 7927}}, {2, {7926, 7927}}, {2, {7928, 7929}}, {2, {7928, 7929}}, {2, {7936, 7944}}, {2, {7937, 7945}}, {2, {7938, 7946}}, {2, {7939, 7947}}, {2, {7940, 7948}}, {2, {7941, 7949}}, {2, {7942, 7950}}, {2, {7943, 7951}}, {2, {7936, 7944}}, {2, {7937, 7945}}, {2, {7938, 7946}}, {2, {7939, 7947}}, {2, {7940, 7948}}, {2, {7941, 7949}}, {2, {7942, 7950}}, {2, {7943, 7951}}, {2, {7952, 7960}}, {2, {7953, 7961}}, {2, {7954, 7962}}, {2, {7955, 7963}}, {2, {7956, 7964}}, {2, {7957, 7965}}, {2, {7952, 7960}}, {2, {7953, 7961}}, {2, {7954, 7962}}, {2, {7955, 7963}}, {2, {7956, 7964}}, {2, {7957, 7965}}, {2, {7968, 7976}}, {2, {7969, 7977}}, {2, {7970, 7978}}, {2, {7971, 7979}}, {2, {7972, 7980}}, {2, {7973, 7981}}, {2, {7974, 7982}}, {2, {7975, 7983}}, {2, {7968, 7976}}, {2, {7969, 7977}}, {2, {7970, 7978}}, {2, {7971, 7979}}, {2, {7972, 7980}}, {2, {7973, 7981}}, {2, {7974, 7982}}, {2, {7975, 7983}}, {2, {7984, 7992}}, {2, {7985, 7993}}, {2, {7986, 7994}}, {2, {7987, 7995}}, {2, {7988, 7996}}, {2, {7989, 7997}}, {2, {7990, 7998}}, {2, {7991, 7999}}, {2, {7984, 7992}}, {2, {7985, 7993}}, {2, {7986, 7994}}, {2, {7987, 7995}}, {2, {7988, 7996}}, {2, {7989, 7997}}, {2, {7990, 7998}}, {2, {7991, 7999}}, {2, {8000, 8008}}, {2, {8001, 8009}}, {2, {8002, 8010}}, {2, {8003, 8011}}, {2, {8004, 8012}}, {2, {8005, 8013}}, {2, {8000, 8008}}, {2, {8001, 8009}}, {2, {8002, 8010}}, {2, {8003, 8011}}, {2, {8004, 8012}}, {2, {8005, 8013}}, {2, {8017, 8025}}, {2, {8019, 8027}}, {2, {8021, 8029}}, {2, {8023, 8031}}, {2, {8017, 8025}}, {2, {8019, 8027}}, {2, {8021, 8029}}, {2, {8023, 8031}}, {2, {8032, 8040}}, {2, {8033, 8041}}, {2, {8034, 8042}}, {2, {8035, 8043}}, {2, {8036, 8044}}, {2, {8037, 8045}}, {2, {8038, 8046}}, {2, {8039, 8047}}, {2, {8032, 8040}}, {2, {8033, 8041}}, {2, {8034, 8042}}, {2, {8035, 8043}}, {2, {8036, 8044}}, {2, {8037, 8045}}, {2, {8038, 8046}}, {2, {8039, 8047}}, {2, {8048, 8122}}, {2, {8049, 8123}}, {2, {8050, 8136}}, {2, {8051, 8137}}, {2, {8052, 8138}}, {2, {8053, 8139}}, {2, {8054, 8154}}, {2, {8055, 8155}}, {2, {8056, 8184}}, {2, {8057, 8185}}, {2, {8058, 8170}}, {2, {8059, 8171}}, {2, {8060, 8186}}, {2, {8061, 8187}}, {2, {8112, 8120}}, {2, {8113, 8121}}, {2, {8112, 8120}}, {2, {8113, 8121}}, {2, {8048, 8122}}, {2, {8049, 8123}}, {4, {837, 921, 953, 8126}}, {2, {8050, 8136}}, {2, {8051, 8137}}, {2, {8052, 8138}}, {2, {8053, 8139}}, {2, {8144, 8152}}, {2, {8145, 8153}}, {2, {8144, 8152}}, {2, {8145, 8153}}, {2, {8054, 8154}}, {2, {8055, 8155}}, {2, {8160, 8168}}, {2, {8161, 8169}}, {2, {8165, 8172}}, {2, {8160, 8168}}, {2, {8161, 8169}}, {2, {8058, 8170}}, {2, {8059, 8171}}, {2, {8165, 8172}}, {2, {8056, 8184}}, {2, {8057, 8185}}, {2, {8060, 8186}}, {2, {8061, 8187}}, {2, {8498, 8526}}, {2, {8498, 8526}}, {2, {8544, 8560}}, {2, {8545, 8561}}, {2, {8546, 8562}}, {2, {8547, 8563}}, {2, {8548, 8564}}, {2, {8549, 8565}}, {2, {8550, 8566}}, {2, {8551, 8567}}, {2, {8552, 8568}}, {2, {8553, 8569}}, {2, {8554, 8570}}, {2, {8555, 8571}}, {2, {8556, 8572}}, {2, {8557, 8573}}, {2, {8558, 8574}}, {2, {8559, 8575}}, {2, {8544, 8560}}, {2, {8545, 8561}}, {2, {8546, 8562}}, {2, {8547, 8563}}, {2, {8548, 8564}}, {2, {8549, 8565}}, {2, {8550, 8566}}, {2, {8551, 8567}}, {2, {8552, 8568}}, {2, {8553, 8569}}, {2, {8554, 8570}}, {2, {8555, 8571}}, {2, {8556, 8572}}, {2, {8557, 8573}}, {2, {8558, 8574}}, {2, {8559, 8575}}, {2, {8579, 8580}}, {2, {8579, 8580}}, {2, {9398, 9424}}, {2, {9399, 9425}}, {2, {9400, 9426}}, {2, {9401, 9427}}, {2, {9402, 9428}}, {2, {9403, 9429}}, {2, {9404, 9430}}, {2, {9405, 9431}}, {2, {9406, 9432}}, {2, {9407, 9433}}, {2, {9408, 9434}}, {2, {9409, 9435}}, {2, {9410, 9436}}, {2, {9411, 9437}}, {2, {9412, 9438}}, {2, {9413, 9439}}, {2, {9414, 9440}}, {2, {9415, 9441}}, {2, {9416, 9442}}, {2, {9417, 9443}}, {2, {9418, 9444}}, {2, {9419, 9445}}, {2, {9420, 9446}}, {2, {9421, 9447}}, {2, {9422, 9448}}, {2, {9423, 9449}}, {2, {9398, 9424}}, {2, {9399, 9425}}, {2, {9400, 9426}}, {2, {9401, 9427}}, {2, {9402, 9428}}, {2, {9403, 9429}}, {2, {9404, 9430}}, {2, {9405, 9431}}, {2, {9406, 9432}}, {2, {9407, 9433}}, {2, {9408, 9434}}, {2, {9409, 9435}}, {2, {9410, 9436}}, {2, {9411, 9437}}, {2, {9412, 9438}}, {2, {9413, 9439}}, {2, {9414, 9440}}, {2, {9415, 9441}}, {2, {9416, 9442}}, {2, {9417, 9443}}, {2, {9418, 9444}}, {2, {9419, 9445}}, {2, {9420, 9446}}, {2, {9421, 9447}}, {2, {9422, 9448}}, {2, {9423, 9449}}, {2, {11264, 11312}}, {2, {11265, 11313}}, {2, {11266, 11314}}, {2, {11267, 11315}}, {2, {11268, 11316}}, {2, {11269, 11317}}, {2, {11270, 11318}}, {2, {11271, 11319}}, {2, {11272, 11320}}, {2, {11273, 11321}}, {2, {11274, 11322}}, {2, {11275, 11323}}, {2, {11276, 11324}}, {2, {11277, 11325}}, {2, {11278, 11326}}, {2, {11279, 11327}}, {2, {11280, 11328}}, {2, {11281, 11329}}, {2, {11282, 11330}}, {2, {11283, 11331}}, {2, {11284, 11332}}, {2, {11285, 11333}}, {2, {11286, 11334}}, {2, {11287, 11335}}, {2, {11288, 11336}}, {2, {11289, 11337}}, {2, {11290, 11338}}, {2, {11291, 11339}}, {2, {11292, 11340}}, {2, {11293, 11341}}, {2, {11294, 11342}}, {2, {11295, 11343}}, {2, {11296, 11344}}, {2, {11297, 11345}}, {2, {11298, 11346}}, {2, {11299, 11347}}, {2, {11300, 11348}}, {2, {11301, 11349}}, {2, {11302, 11350}}, {2, {11303, 11351}}, {2, {11304, 11352}}, {2, {11305, 11353}}, {2, {11306, 11354}}, {2, {11307, 11355}}, {2, {11308, 11356}}, {2, {11309, 11357}}, {2, {11310, 11358}}, {2, {11264, 11312}}, {2, {11265, 11313}}, {2, {11266, 11314}}, {2, {11267, 11315}}, {2, {11268, 11316}}, {2, {11269, 11317}}, {2, {11270, 11318}}, {2, {11271, 11319}}, {2, {11272, 11320}}, {2, {11273, 11321}}, {2, {11274, 11322}}, {2, {11275, 11323}}, {2, {11276, 11324}}, {2, {11277, 11325}}, {2, {11278, 11326}}, {2, {11279, 11327}}, {2, {11280, 11328}}, {2, {11281, 11329}}, {2, {11282, 11330}}, {2, {11283, 11331}}, {2, {11284, 11332}}, {2, {11285, 11333}}, {2, {11286, 11334}}, {2, {11287, 11335}}, {2, {11288, 11336}}, {2, {11289, 11337}}, {2, {11290, 11338}}, {2, {11291, 11339}}, {2, {11292, 11340}}, {2, {11293, 11341}}, {2, {11294, 11342}}, {2, {11295, 11343}}, {2, {11296, 11344}}, {2, {11297, 11345}}, {2, {11298, 11346}}, {2, {11299, 11347}}, {2, {11300, 11348}}, {2, {11301, 11349}}, {2, {11302, 11350}}, {2, {11303, 11351}}, {2, {11304, 11352}}, {2, {11305, 11353}}, {2, {11306, 11354}}, {2, {11307, 11355}}, {2, {11308, 11356}}, {2, {11309, 11357}}, {2, {11310, 11358}}, {2, {11360, 11361}}, {2, {11360, 11361}}, {2, {619, 11362}}, {2, {7549, 11363}}, {2, {637, 11364}}, {2, {570, 11365}}, {2, {574, 11366}}, {2, {11367, 11368}}, {2, {11367, 11368}}, {2, {11369, 11370}}, {2, {11369, 11370}}, {2, {11371, 11372}}, {2, {11371, 11372}}, {2, {11381, 11382}}, {2, {11381, 11382}}, {2, {11392, 11393}}, {2, {11392, 11393}}, {2, {11394, 11395}}, {2, {11394, 11395}}, {2, {11396, 11397}}, {2, {11396, 11397}}, {2, {11398, 11399}}, {2, {11398, 11399}}, {2, {11400, 11401}}, {2, {11400, 11401}}, {2, {11402, 11403}}, {2, {11402, 11403}}, {2, {11404, 11405}}, {2, {11404, 11405}}, {2, {11406, 11407}}, {2, {11406, 11407}}, {2, {11408, 11409}}, {2, {11408, 11409}}, {2, {11410, 11411}}, {2, {11410, 11411}}, {2, {11412, 11413}}, {2, {11412, 11413}}, {2, {11414, 11415}}, {2, {11414, 11415}}, {2, {11416, 11417}}, {2, {11416, 11417}}, {2, {11418, 11419}}, {2, {11418, 11419}}, {2, {11420, 11421}}, {2, {11420, 11421}}, {2, {11422, 11423}}, {2, {11422, 11423}}, {2, {11424, 11425}}, {2, {11424, 11425}}, {2, {11426, 11427}}, {2, {11426, 11427}}, {2, {11428, 11429}}, {2, {11428, 11429}}, {2, {11430, 11431}}, {2, {11430, 11431}}, {2, {11432, 11433}}, {2, {11432, 11433}}, {2, {11434, 11435}}, {2, {11434, 11435}}, {2, {11436, 11437}}, {2, {11436, 11437}}, {2, {11438, 11439}}, {2, {11438, 11439}}, {2, {11440, 11441}}, {2, {11440, 11441}}, {2, {11442, 11443}}, {2, {11442, 11443}}, {2, {11444, 11445}}, {2, {11444, 11445}}, {2, {11446, 11447}}, {2, {11446, 11447}}, {2, {11448, 11449}}, {2, {11448, 11449}}, {2, {11450, 11451}}, {2, {11450, 11451}}, {2, {11452, 11453}}, {2, {11452, 11453}}, {2, {11454, 11455}}, {2, {11454, 11455}}, {2, {11456, 11457}}, {2, {11456, 11457}}, {2, {11458, 11459}}, {2, {11458, 11459}}, {2, {11460, 11461}}, {2, {11460, 11461}}, {2, {11462, 11463}}, {2, {11462, 11463}}, {2, {11464, 11465}}, {2, {11464, 11465}}, {2, {11466, 11467}}, {2, {11466, 11467}}, {2, {11468, 11469}}, {2, {11468, 11469}}, {2, {11470, 11471}}, {2, {11470, 11471}}, {2, {11472, 11473}}, {2, {11472, 11473}}, {2, {11474, 11475}}, {2, {11474, 11475}}, {2, {11476, 11477}}, {2, {11476, 11477}}, {2, {11478, 11479}}, {2, {11478, 11479}}, {2, {11480, 11481}}, {2, {11480, 11481}}, {2, {11482, 11483}}, {2, {11482, 11483}}, {2, {11484, 11485}}, {2, {11484, 11485}}, {2, {11486, 11487}}, {2, {11486, 11487}}, {2, {11488, 11489}}, {2, {11488, 11489}}, {2, {11490, 11491}}, {2, {11490, 11491}}, {2, {4256, 11520}}, {2, {4257, 11521}}, {2, {4258, 11522}}, {2, {4259, 11523}}, {2, {4260, 11524}}, {2, {4261, 11525}}, {2, {4262, 11526}}, {2, {4263, 11527}}, {2, {4264, 11528}}, {2, {4265, 11529}}, {2, {4266, 11530}}, {2, {4267, 11531}}, {2, {4268, 11532}}, {2, {4269, 11533}}, {2, {4270, 11534}}, {2, {4271, 11535}}, {2, {4272, 11536}}, {2, {4273, 11537}}, {2, {4274, 11538}}, {2, {4275, 11539}}, {2, {4276, 11540}}, {2, {4277, 11541}}, {2, {4278, 11542}}, {2, {4279, 11543}}, {2, {4280, 11544}}, {2, {4281, 11545}}, {2, {4282, 11546}}, {2, {4283, 11547}}, {2, {4284, 11548}}, {2, {4285, 11549}}, {2, {4286, 11550}}, {2, {4287, 11551}}, {2, {4288, 11552}}, {2, {4289, 11553}}, {2, {4290, 11554}}, {2, {4291, 11555}}, {2, {4292, 11556}}, {2, {4293, 11557}}, {0, {0}} }; // NOLINT
static const uint16_t kEcma262UnCanonicalizeTable0Size = 1656;
static const int32_t kEcma262UnCanonicalizeTable0[3312] = { 65, 1, 66, 5, 67, 9, 68, 13, 69, 17, 70, 21, 71, 25, 72, 29, 73, 33, 74, 37, 75, 41, 76, 45, 77, 49, 78, 53, 79, 57, 80, 61, 81, 65, 82, 69, 83, 73, 84, 77, 85, 81, 86, 85, 87, 89, 88, 93, 89, 97, 90, 101, 97, 105, 98, 109, 99, 113, 100, 117, 101, 121, 102, 125, 103, 129, 104, 133, 105, 137, 106, 141, 107, 145, 108, 149, 109, 153, 110, 157, 111, 161, 112, 165, 113, 169, 114, 173, 115, 177, 116, 181, 117, 185, 118, 189, 119, 193, 120, 197, 121, 201, 122, 205, 181, 209, 192, 213, 193, 217, 194, 221, 195, 225, 196, 229, 197, 233, 198, 237, 199, 241, 200, 245, 201, 249, 202, 253, 203, 257, 204, 261, 205, 265, 206, 269, 207, 273, 208, 277, 209, 281, 210, 285, 211, 289, 212, 293, 213, 297, 214, 301, 216, 305, 217, 309, 218, 313, 219, 317, 220, 321, 221, 325, 222, 329, 224, 333, 225, 337, 226, 341, 227, 345, 228, 349, 229, 353, 230, 357, 231, 361, 232, 365, 233, 369, 234, 373, 235, 377, 236, 381, 237, 385, 238, 389, 239, 393, 240, 397, 241, 401, 242, 405, 243, 409, 244, 413, 245, 417, 246, 421, 248, 425, 249, 429, 250, 433, 251, 437, 252, 441, 253, 445, 254, 449, 255, 453, 256, 457, 257, 461, 258, 465, 259, 469, 260, 473, 261, 477, 262, 481, 263, 485, 264, 489, 265, 493, 266, 497, 267, 501, 268, 505, 269, 509, 270, 513, 271, 517, 272, 521, 273, 525, 274, 529, 275, 533, 276, 537, 277, 541, 278, 545, 279, 549, 280, 553, 281, 557, 282, 561, 283, 565, 284, 569, 285, 573, 286, 577, 287, 581, 288, 585, 289, 589, 290, 593, 291, 597, 292, 601, 293, 605, 294, 609, 295, 613, 296, 617, 297, 621, 298, 625, 299, 629, 300, 633, 301, 637, 302, 641, 303, 645, 306, 649, 307, 653, 308, 657, 309, 661, 310, 665, 311, 669, 313, 673, 314, 677, 315, 681, 316, 685, 317, 689, 318, 693, 319, 697, 320, 701, 321, 705, 322, 709, 323, 713, 324, 717, 325, 721, 326, 725, 327, 729, 328, 733, 330, 737, 331, 741, 332, 745, 333, 749, 334, 753, 335, 757, 336, 761, 337, 765, 338, 769, 339, 773, 340, 777, 341, 781, 342, 785, 343, 789, 344, 793, 345, 797, 346, 801, 347, 805, 348, 809, 349, 813, 350, 817, 351, 821, 352, 825, 353, 829, 354, 833, 355, 837, 356, 841, 357, 845, 358, 849, 359, 853, 360, 857, 361, 861, 362, 865, 363, 869, 364, 873, 365, 877, 366, 881, 367, 885, 368, 889, 369, 893, 370, 897, 371, 901, 372, 905, 373, 909, 374, 913, 375, 917, 376, 921, 377, 925, 378, 929, 379, 933, 380, 937, 381, 941, 382, 945, 384, 949, 385, 953, 386, 957, 387, 961, 388, 965, 389, 969, 390, 973, 391, 977, 392, 981, 393, 985, 394, 989, 395, 993, 396, 997, 398, 1001, 399, 1005, 400, 1009, 401, 1013, 402, 1017, 403, 1021, 404, 1025, 405, 1029, 406, 1033, 407, 1037, 408, 1041, 409, 1045, 410, 1049, 412, 1053, 413, 1057, 414, 1061, 415, 1065, 416, 1069, 417, 1073, 418, 1077, 419, 1081, 420, 1085, 421, 1089, 422, 1093, 423, 1097, 424, 1101, 425, 1105, 428, 1109, 429, 1113, 430, 1117, 431, 1121, 432, 1125, 433, 1129, 434, 1133, 435, 1137, 436, 1141, 437, 1145, 438, 1149, 439, 1153, 440, 1157, 441, 1161, 444, 1165, 445, 1169, 447, 1173, 452, 1177, 453, 1181, 454, 1185, 455, 1189, 456, 1193, 457, 1197, 458, 1201, 459, 1205, 460, 1209, 461, 1213, 462, 1217, 463, 1221, 464, 1225, 465, 1229, 466, 1233, 467, 1237, 468, 1241, 469, 1245, 470, 1249, 471, 1253, 472, 1257, 473, 1261, 474, 1265, 475, 1269, 476, 1273, 477, 1277, 478, 1281, 479, 1285, 480, 1289, 481, 1293, 482, 1297, 483, 1301, 484, 1305, 485, 1309, 486, 1313, 487, 1317, 488, 1321, 489, 1325, 490, 1329, 491, 1333, 492, 1337, 493, 1341, 494, 1345, 495, 1349, 497, 1353, 498, 1357, 499, 1361, 500, 1365, 501, 1369, 502, 1373, 503, 1377, 504, 1381, 505, 1385, 506, 1389, 507, 1393, 508, 1397, 509, 1401, 510, 1405, 511, 1409, 512, 1413, 513, 1417, 514, 1421, 515, 1425, 516, 1429, 517, 1433, 518, 1437, 519, 1441, 520, 1445, 521, 1449, 522, 1453, 523, 1457, 524, 1461, 525, 1465, 526, 1469, 527, 1473, 528, 1477, 529, 1481, 530, 1485, 531, 1489, 532, 1493, 533, 1497, 534, 1501, 535, 1505, 536, 1509, 537, 1513, 538, 1517, 539, 1521, 540, 1525, 541, 1529, 542, 1533, 543, 1537, 544, 1541, 546, 1545, 547, 1549, 548, 1553, 549, 1557, 550, 1561, 551, 1565, 552, 1569, 553, 1573, 554, 1577, 555, 1581, 556, 1585, 557, 1589, 558, 1593, 559, 1597, 560, 1601, 561, 1605, 562, 1609, 563, 1613, 570, 1617, 571, 1621, 572, 1625, 573, 1629, 574, 1633, 577, 1637, 578, 1641, 579, 1645, 580, 1649, 581, 1653, 582, 1657, 583, 1661, 584, 1665, 585, 1669, 586, 1673, 587, 1677, 588, 1681, 589, 1685, 590, 1689, 591, 1693, 595, 1697, 596, 1701, 598, 1705, 599, 1709, 601, 1713, 603, 1717, 608, 1721, 611, 1725, 616, 1729, 617, 1733, 619, 1737, 623, 1741, 626, 1745, 629, 1749, 637, 1753, 640, 1757, 643, 1761, 648, 1765, 649, 1769, 650, 1773, 651, 1777, 652, 1781, 658, 1785, 837, 1789, 891, 1793, 892, 1797, 893, 1801, 902, 1805, 904, 1809, 905, 1813, 906, 1817, 908, 1821, 910, 1825, 911, 1829, 913, 1833, 914, 1837, 915, 1841, 916, 1845, 917, 1849, 918, 1853, 919, 1857, 920, 1861, 921, 1865, 922, 1869, 923, 1873, 924, 1877, 925, 1881, 926, 1885, 927, 1889, 928, 1893, 929, 1897, 931, 1901, 932, 1905, 933, 1909, 934, 1913, 935, 1917, 936, 1921, 937, 1925, 938, 1929, 939, 1933, 940, 1937, 941, 1941, 942, 1945, 943, 1949, 945, 1953, 946, 1957, 947, 1961, 948, 1965, 949, 1969, 950, 1973, 951, 1977, 952, 1981, 953, 1985, 954, 1989, 955, 1993, 956, 1997, 957, 2001, 958, 2005, 959, 2009, 960, 2013, 961, 2017, 962, 2021, 963, 2025, 964, 2029, 965, 2033, 966, 2037, 967, 2041, 968, 2045, 969, 2049, 970, 2053, 971, 2057, 972, 2061, 973, 2065, 974, 2069, 976, 2073, 977, 2077, 981, 2081, 982, 2085, 984, 2089, 985, 2093, 986, 2097, 987, 2101, 988, 2105, 989, 2109, 990, 2113, 991, 2117, 992, 2121, 993, 2125, 994, 2129, 995, 2133, 996, 2137, 997, 2141, 998, 2145, 999, 2149, 1000, 2153, 1001, 2157, 1002, 2161, 1003, 2165, 1004, 2169, 1005, 2173, 1006, 2177, 1007, 2181, 1008, 2185, 1009, 2189, 1010, 2193, 1013, 2197, 1015, 2201, 1016, 2205, 1017, 2209, 1018, 2213, 1019, 2217, 1021, 2221, 1022, 2225, 1023, 2229, 1024, 2233, 1025, 2237, 1026, 2241, 1027, 2245, 1028, 2249, 1029, 2253, 1030, 2257, 1031, 2261, 1032, 2265, 1033, 2269, 1034, 2273, 1035, 2277, 1036, 2281, 1037, 2285, 1038, 2289, 1039, 2293, 1040, 2297, 1041, 2301, 1042, 2305, 1043, 2309, 1044, 2313, 1045, 2317, 1046, 2321, 1047, 2325, 1048, 2329, 1049, 2333, 1050, 2337, 1051, 2341, 1052, 2345, 1053, 2349, 1054, 2353, 1055, 2357, 1056, 2361, 1057, 2365, 1058, 2369, 1059, 2373, 1060, 2377, 1061, 2381, 1062, 2385, 1063, 2389, 1064, 2393, 1065, 2397, 1066, 2401, 1067, 2405, 1068, 2409, 1069, 2413, 1070, 2417, 1071, 2421, 1072, 2425, 1073, 2429, 1074, 2433, 1075, 2437, 1076, 2441, 1077, 2445, 1078, 2449, 1079, 2453, 1080, 2457, 1081, 2461, 1082, 2465, 1083, 2469, 1084, 2473, 1085, 2477, 1086, 2481, 1087, 2485, 1088, 2489, 1089, 2493, 1090, 2497, 1091, 2501, 1092, 2505, 1093, 2509, 1094, 2513, 1095, 2517, 1096, 2521, 1097, 2525, 1098, 2529, 1099, 2533, 1100, 2537, 1101, 2541, 1102, 2545, 1103, 2549, 1104, 2553, 1105, 2557, 1106, 2561, 1107, 2565, 1108, 2569, 1109, 2573, 1110, 2577, 1111, 2581, 1112, 2585, 1113, 2589, 1114, 2593, 1115, 2597, 1116, 2601, 1117, 2605, 1118, 2609, 1119, 2613, 1120, 2617, 1121, 2621, 1122, 2625, 1123, 2629, 1124, 2633, 1125, 2637, 1126, 2641, 1127, 2645, 1128, 2649, 1129, 2653, 1130, 2657, 1131, 2661, 1132, 2665, 1133, 2669, 1134, 2673, 1135, 2677, 1136, 2681, 1137, 2685, 1138, 2689, 1139, 2693, 1140, 2697, 1141, 2701, 1142, 2705, 1143, 2709, 1144, 2713, 1145, 2717, 1146, 2721, 1147, 2725, 1148, 2729, 1149, 2733, 1150, 2737, 1151, 2741, 1152, 2745, 1153, 2749, 1162, 2753, 1163, 2757, 1164, 2761, 1165, 2765, 1166, 2769, 1167, 2773, 1168, 2777, 1169, 2781, 1170, 2785, 1171, 2789, 1172, 2793, 1173, 2797, 1174, 2801, 1175, 2805, 1176, 2809, 1177, 2813, 1178, 2817, 1179, 2821, 1180, 2825, 1181, 2829, 1182, 2833, 1183, 2837, 1184, 2841, 1185, 2845, 1186, 2849, 1187, 2853, 1188, 2857, 1189, 2861, 1190, 2865, 1191, 2869, 1192, 2873, 1193, 2877, 1194, 2881, 1195, 2885, 1196, 2889, 1197, 2893, 1198, 2897, 1199, 2901, 1200, 2905, 1201, 2909, 1202, 2913, 1203, 2917, 1204, 2921, 1205, 2925, 1206, 2929, 1207, 2933, 1208, 2937, 1209, 2941, 1210, 2945, 1211, 2949, 1212, 2953, 1213, 2957, 1214, 2961, 1215, 2965, 1216, 2969, 1217, 2973, 1218, 2977, 1219, 2981, 1220, 2985, 1221, 2989, 1222, 2993, 1223, 2997, 1224, 3001, 1225, 3005, 1226, 3009, 1227, 3013, 1228, 3017, 1229, 3021, 1230, 3025, 1231, 3029, 1232, 3033, 1233, 3037, 1234, 3041, 1235, 3045, 1236, 3049, 1237, 3053, 1238, 3057, 1239, 3061, 1240, 3065, 1241, 3069, 1242, 3073, 1243, 3077, 1244, 3081, 1245, 3085, 1246, 3089, 1247, 3093, 1248, 3097, 1249, 3101, 1250, 3105, 1251, 3109, 1252, 3113, 1253, 3117, 1254, 3121, 1255, 3125, 1256, 3129, 1257, 3133, 1258, 3137, 1259, 3141, 1260, 3145, 1261, 3149, 1262, 3153, 1263, 3157, 1264, 3161, 1265, 3165, 1266, 3169, 1267, 3173, 1268, 3177, 1269, 3181, 1270, 3185, 1271, 3189, 1272, 3193, 1273, 3197, 1274, 3201, 1275, 3205, 1276, 3209, 1277, 3213, 1278, 3217, 1279, 3221, 1280, 3225, 1281, 3229, 1282, 3233, 1283, 3237, 1284, 3241, 1285, 3245, 1286, 3249, 1287, 3253, 1288, 3257, 1289, 3261, 1290, 3265, 1291, 3269, 1292, 3273, 1293, 3277, 1294, 3281, 1295, 3285, 1296, 3289, 1297, 3293, 1298, 3297, 1299, 3301, 1329, 3305, 1330, 3309, 1331, 3313, 1332, 3317, 1333, 3321, 1334, 3325, 1335, 3329, 1336, 3333, 1337, 3337, 1338, 3341, 1339, 3345, 1340, 3349, 1341, 3353, 1342, 3357, 1343, 3361, 1344, 3365, 1345, 3369, 1346, 3373, 1347, 3377, 1348, 3381, 1349, 3385, 1350, 3389, 1351, 3393, 1352, 3397, 1353, 3401, 1354, 3405, 1355, 3409, 1356, 3413, 1357, 3417, 1358, 3421, 1359, 3425, 1360, 3429, 1361, 3433, 1362, 3437, 1363, 3441, 1364, 3445, 1365, 3449, 1366, 3453, 1377, 3457, 1378, 3461, 1379, 3465, 1380, 3469, 1381, 3473, 1382, 3477, 1383, 3481, 1384, 3485, 1385, 3489, 1386, 3493, 1387, 3497, 1388, 3501, 1389, 3505, 1390, 3509, 1391, 3513, 1392, 3517, 1393, 3521, 1394, 3525, 1395, 3529, 1396, 3533, 1397, 3537, 1398, 3541, 1399, 3545, 1400, 3549, 1401, 3553, 1402, 3557, 1403, 3561, 1404, 3565, 1405, 3569, 1406, 3573, 1407, 3577, 1408, 3581, 1409, 3585, 1410, 3589, 1411, 3593, 1412, 3597, 1413, 3601, 1414, 3605, 4256, 3609, 4257, 3613, 4258, 3617, 4259, 3621, 4260, 3625, 4261, 3629, 4262, 3633, 4263, 3637, 4264, 3641, 4265, 3645, 4266, 3649, 4267, 3653, 4268, 3657, 4269, 3661, 4270, 3665, 4271, 3669, 4272, 3673, 4273, 3677, 4274, 3681, 4275, 3685, 4276, 3689, 4277, 3693, 4278, 3697, 4279, 3701, 4280, 3705, 4281, 3709, 4282, 3713, 4283, 3717, 4284, 3721, 4285, 3725, 4286, 3729, 4287, 3733, 4288, 3737, 4289, 3741, 4290, 3745, 4291, 3749, 4292, 3753, 4293, 3757, 7549, 3761, 7680, 3765, 7681, 3769, 7682, 3773, 7683, 3777, 7684, 3781, 7685, 3785, 7686, 3789, 7687, 3793, 7688, 3797, 7689, 3801, 7690, 3805, 7691, 3809, 7692, 3813, 7693, 3817, 7694, 3821, 7695, 3825, 7696, 3829, 7697, 3833, 7698, 3837, 7699, 3841, 7700, 3845, 7701, 3849, 7702, 3853, 7703, 3857, 7704, 3861, 7705, 3865, 7706, 3869, 7707, 3873, 7708, 3877, 7709, 3881, 7710, 3885, 7711, 3889, 7712, 3893, 7713, 3897, 7714, 3901, 7715, 3905, 7716, 3909, 7717, 3913, 7718, 3917, 7719, 3921, 7720, 3925, 7721, 3929, 7722, 3933, 7723, 3937, 7724, 3941, 7725, 3945, 7726, 3949, 7727, 3953, 7728, 3957, 7729, 3961, 7730, 3965, 7731, 3969, 7732, 3973, 7733, 3977, 7734, 3981, 7735, 3985, 7736, 3989, 7737, 3993, 7738, 3997, 7739, 4001, 7740, 4005, 7741, 4009, 7742, 4013, 7743, 4017, 7744, 4021, 7745, 4025, 7746, 4029, 7747, 4033, 7748, 4037, 7749, 4041, 7750, 4045, 7751, 4049, 7752, 4053, 7753, 4057, 7754, 4061, 7755, 4065, 7756, 4069, 7757, 4073, 7758, 4077, 7759, 4081, 7760, 4085, 7761, 4089, 7762, 4093, 7763, 4097, 7764, 4101, 7765, 4105, 7766, 4109, 7767, 4113, 7768, 4117, 7769, 4121, 7770, 4125, 7771, 4129, 7772, 4133, 7773, 4137, 7774, 4141, 7775, 4145, 7776, 4149, 7777, 4153, 7778, 4157, 7779, 4161, 7780, 4165, 7781, 4169, 7782, 4173, 7783, 4177, 7784, 4181, 7785, 4185, 7786, 4189, 7787, 4193, 7788, 4197, 7789, 4201, 7790, 4205, 7791, 4209, 7792, 4213, 7793, 4217, 7794, 4221, 7795, 4225, 7796, 4229, 7797, 4233, 7798, 4237, 7799, 4241, 7800, 4245, 7801, 4249, 7802, 4253, 7803, 4257, 7804, 4261, 7805, 4265, 7806, 4269, 7807, 4273, 7808, 4277, 7809, 4281, 7810, 4285, 7811, 4289, 7812, 4293, 7813, 4297, 7814, 4301, 7815, 4305, 7816, 4309, 7817, 4313, 7818, 4317, 7819, 4321, 7820, 4325, 7821, 4329, 7822, 4333, 7823, 4337, 7824, 4341, 7825, 4345, 7826, 4349, 7827, 4353, 7828, 4357, 7829, 4361, 7835, 4365, 7840, 4369, 7841, 4373, 7842, 4377, 7843, 4381, 7844, 4385, 7845, 4389, 7846, 4393, 7847, 4397, 7848, 4401, 7849, 4405, 7850, 4409, 7851, 4413, 7852, 4417, 7853, 4421, 7854, 4425, 7855, 4429, 7856, 4433, 7857, 4437, 7858, 4441, 7859, 4445, 7860, 4449, 7861, 4453, 7862, 4457, 7863, 4461, 7864, 4465, 7865, 4469, 7866, 4473, 7867, 4477, 7868, 4481, 7869, 4485, 7870, 4489, 7871, 4493, 7872, 4497, 7873, 4501, 7874, 4505, 7875, 4509, 7876, 4513, 7877, 4517, 7878, 4521, 7879, 4525, 7880, 4529, 7881, 4533, 7882, 4537, 7883, 4541, 7884, 4545, 7885, 4549, 7886, 4553, 7887, 4557, 7888, 4561, 7889, 4565, 7890, 4569, 7891, 4573, 7892, 4577, 7893, 4581, 7894, 4585, 7895, 4589, 7896, 4593, 7897, 4597, 7898, 4601, 7899, 4605, 7900, 4609, 7901, 4613, 7902, 4617, 7903, 4621, 7904, 4625, 7905, 4629, 7906, 4633, 7907, 4637, 7908, 4641, 7909, 4645, 7910, 4649, 7911, 4653, 7912, 4657, 7913, 4661, 7914, 4665, 7915, 4669, 7916, 4673, 7917, 4677, 7918, 4681, 7919, 4685, 7920, 4689, 7921, 4693, 7922, 4697, 7923, 4701, 7924, 4705, 7925, 4709, 7926, 4713, 7927, 4717, 7928, 4721, 7929, 4725, 7936, 4729, 7937, 4733, 7938, 4737, 7939, 4741, 7940, 4745, 7941, 4749, 7942, 4753, 7943, 4757, 7944, 4761, 7945, 4765, 7946, 4769, 7947, 4773, 7948, 4777, 7949, 4781, 7950, 4785, 7951, 4789, 7952, 4793, 7953, 4797, 7954, 4801, 7955, 4805, 7956, 4809, 7957, 4813, 7960, 4817, 7961, 4821, 7962, 4825, 7963, 4829, 7964, 4833, 7965, 4837, 7968, 4841, 7969, 4845, 7970, 4849, 7971, 4853, 7972, 4857, 7973, 4861, 7974, 4865, 7975, 4869, 7976, 4873, 7977, 4877, 7978, 4881, 7979, 4885, 7980, 4889, 7981, 4893, 7982, 4897, 7983, 4901, 7984, 4905, 7985, 4909, 7986, 4913, 7987, 4917, 7988, 4921, 7989, 4925, 7990, 4929, 7991, 4933, 7992, 4937, 7993, 4941, 7994, 4945, 7995, 4949, 7996, 4953, 7997, 4957, 7998, 4961, 7999, 4965, 8000, 4969, 8001, 4973, 8002, 4977, 8003, 4981, 8004, 4985, 8005, 4989, 8008, 4993, 8009, 4997, 8010, 5001, 8011, 5005, 8012, 5009, 8013, 5013, 8017, 5017, 8019, 5021, 8021, 5025, 8023, 5029, 8025, 5033, 8027, 5037, 8029, 5041, 8031, 5045, 8032, 5049, 8033, 5053, 8034, 5057, 8035, 5061, 8036, 5065, 8037, 5069, 8038, 5073, 8039, 5077, 8040, 5081, 8041, 5085, 8042, 5089, 8043, 5093, 8044, 5097, 8045, 5101, 8046, 5105, 8047, 5109, 8048, 5113, 8049, 5117, 8050, 5121, 8051, 5125, 8052, 5129, 8053, 5133, 8054, 5137, 8055, 5141, 8056, 5145, 8057, 5149, 8058, 5153, 8059, 5157, 8060, 5161, 8061, 5165, 8112, 5169, 8113, 5173, 8120, 5177, 8121, 5181, 8122, 5185, 8123, 5189, 8126, 5193, 8136, 5197, 8137, 5201, 8138, 5205, 8139, 5209, 8144, 5213, 8145, 5217, 8152, 5221, 8153, 5225, 8154, 5229, 8155, 5233, 8160, 5237, 8161, 5241, 8165, 5245, 8168, 5249, 8169, 5253, 8170, 5257, 8171, 5261, 8172, 5265, 8184, 5269, 8185, 5273, 8186, 5277, 8187, 5281, 8498, 5285, 8526, 5289, 8544, 5293, 8545, 5297, 8546, 5301, 8547, 5305, 8548, 5309, 8549, 5313, 8550, 5317, 8551, 5321, 8552, 5325, 8553, 5329, 8554, 5333, 8555, 5337, 8556, 5341, 8557, 5345, 8558, 5349, 8559, 5353, 8560, 5357, 8561, 5361, 8562, 5365, 8563, 5369, 8564, 5373, 8565, 5377, 8566, 5381, 8567, 5385, 8568, 5389, 8569, 5393, 8570, 5397, 8571, 5401, 8572, 5405, 8573, 5409, 8574, 5413, 8575, 5417, 8579, 5421, 8580, 5425, 9398, 5429, 9399, 5433, 9400, 5437, 9401, 5441, 9402, 5445, 9403, 5449, 9404, 5453, 9405, 5457, 9406, 5461, 9407, 5465, 9408, 5469, 9409, 5473, 9410, 5477, 9411, 5481, 9412, 5485, 9413, 5489, 9414, 5493, 9415, 5497, 9416, 5501, 9417, 5505, 9418, 5509, 9419, 5513, 9420, 5517, 9421, 5521, 9422, 5525, 9423, 5529, 9424, 5533, 9425, 5537, 9426, 5541, 9427, 5545, 9428, 5549, 9429, 5553, 9430, 5557, 9431, 5561, 9432, 5565, 9433, 5569, 9434, 5573, 9435, 5577, 9436, 5581, 9437, 5585, 9438, 5589, 9439, 5593, 9440, 5597, 9441, 5601, 9442, 5605, 9443, 5609, 9444, 5613, 9445, 5617, 9446, 5621, 9447, 5625, 9448, 5629, 9449, 5633, 11264, 5637, 11265, 5641, 11266, 5645, 11267, 5649, 11268, 5653, 11269, 5657, 11270, 5661, 11271, 5665, 11272, 5669, 11273, 5673, 11274, 5677, 11275, 5681, 11276, 5685, 11277, 5689, 11278, 5693, 11279, 5697, 11280, 5701, 11281, 5705, 11282, 5709, 11283, 5713, 11284, 5717, 11285, 5721, 11286, 5725, 11287, 5729, 11288, 5733, 11289, 5737, 11290, 5741, 11291, 5745, 11292, 5749, 11293, 5753, 11294, 5757, 11295, 5761, 11296, 5765, 11297, 5769, 11298, 5773, 11299, 5777, 11300, 5781, 11301, 5785, 11302, 5789, 11303, 5793, 11304, 5797, 11305, 5801, 11306, 5805, 11307, 5809, 11308, 5813, 11309, 5817, 11310, 5821, 11312, 5825, 11313, 5829, 11314, 5833, 11315, 5837, 11316, 5841, 11317, 5845, 11318, 5849, 11319, 5853, 11320, 5857, 11321, 5861, 11322, 5865, 11323, 5869, 11324, 5873, 11325, 5877, 11326, 5881, 11327, 5885, 11328, 5889, 11329, 5893, 11330, 5897, 11331, 5901, 11332, 5905, 11333, 5909, 11334, 5913, 11335, 5917, 11336, 5921, 11337, 5925, 11338, 5929, 11339, 5933, 11340, 5937, 11341, 5941, 11342, 5945, 11343, 5949, 11344, 5953, 11345, 5957, 11346, 5961, 11347, 5965, 11348, 5969, 11349, 5973, 11350, 5977, 11351, 5981, 11352, 5985, 11353, 5989, 11354, 5993, 11355, 5997, 11356, 6001, 11357, 6005, 11358, 6009, 11360, 6013, 11361, 6017, 11362, 6021, 11363, 6025, 11364, 6029, 11365, 6033, 11366, 6037, 11367, 6041, 11368, 6045, 11369, 6049, 11370, 6053, 11371, 6057, 11372, 6061, 11381, 6065, 11382, 6069, 11392, 6073, 11393, 6077, 11394, 6081, 11395, 6085, 11396, 6089, 11397, 6093, 11398, 6097, 11399, 6101, 11400, 6105, 11401, 6109, 11402, 6113, 11403, 6117, 11404, 6121, 11405, 6125, 11406, 6129, 11407, 6133, 11408, 6137, 11409, 6141, 11410, 6145, 11411, 6149, 11412, 6153, 11413, 6157, 11414, 6161, 11415, 6165, 11416, 6169, 11417, 6173, 11418, 6177, 11419, 6181, 11420, 6185, 11421, 6189, 11422, 6193, 11423, 6197, 11424, 6201, 11425, 6205, 11426, 6209, 11427, 6213, 11428, 6217, 11429, 6221, 11430, 6225, 11431, 6229, 11432, 6233, 11433, 6237, 11434, 6241, 11435, 6245, 11436, 6249, 11437, 6253, 11438, 6257, 11439, 6261, 11440, 6265, 11441, 6269, 11442, 6273, 11443, 6277, 11444, 6281, 11445, 6285, 11446, 6289, 11447, 6293, 11448, 6297, 11449, 6301, 11450, 6305, 11451, 6309, 11452, 6313, 11453, 6317, 11454, 6321, 11455, 6325, 11456, 6329, 11457, 6333, 11458, 6337, 11459, 6341, 11460, 6345, 11461, 6349, 11462, 6353, 11463, 6357, 11464, 6361, 11465, 6365, 11466, 6369, 11467, 6373, 11468, 6377, 11469, 6381, 11470, 6385, 11471, 6389, 11472, 6393, 11473, 6397, 11474, 6401, 11475, 6405, 11476, 6409, 11477, 6413, 11478, 6417, 11479, 6421, 11480, 6425, 11481, 6429, 11482, 6433, 11483, 6437, 11484, 6441, 11485, 6445, 11486, 6449, 11487, 6453, 11488, 6457, 11489, 6461, 11490, 6465, 11491, 6469, 11520, 6473, 11521, 6477, 11522, 6481, 11523, 6485, 11524, 6489, 11525, 6493, 11526, 6497, 11527, 6501, 11528, 6505, 11529, 6509, 11530, 6513, 11531, 6517, 11532, 6521, 11533, 6525, 11534, 6529, 11535, 6533, 11536, 6537, 11537, 6541, 11538, 6545, 11539, 6549, 11540, 6553, 11541, 6557, 11542, 6561, 11543, 6565, 11544, 6569, 11545, 6573, 11546, 6577, 11547, 6581, 11548, 6585, 11549, 6589, 11550, 6593, 11551, 6597, 11552, 6601, 11553, 6605, 11554, 6609, 11555, 6613, 11556, 6617, 11557, 6621 }; // NOLINT
static const MultiCharacterSpecialCase<4> kEcma262UnCanonicalizeMultiStrings1[] = { {2, {65313, 65345}}, {2, {65314, 65346}}, {2, {65315, 65347}}, {2, {65316, 65348}}, {2, {65317, 65349}}, {2, {65318, 65350}}, {2, {65319, 65351}}, {2, {65320, 65352}}, {2, {65321, 65353}}, {2, {65322, 65354}}, {2, {65323, 65355}}, {2, {65324, 65356}}, {2, {65325, 65357}}, {2, {65326, 65358}}, {2, {65327, 65359}}, {2, {65328, 65360}}, {2, {65329, 65361}}, {2, {65330, 65362}}, {2, {65331, 65363}}, {2, {65332, 65364}}, {2, {65333, 65365}}, {2, {65334, 65366}}, {2, {65335, 65367}}, {2, {65336, 65368}}, {2, {65337, 65369}}, {2, {65338, 65370}}, {2, {65313, 65345}}, {2, {65314, 65346}}, {2, {65315, 65347}}, {2, {65316, 65348}}, {2, {65317, 65349}}, {2, {65318, 65350}}, {2, {65319, 65351}}, {2, {65320, 65352}}, {2, {65321, 65353}}, {2, {65322, 65354}}, {2, {65323, 65355}}, {2, {65324, 65356}}, {2, {65325, 65357}}, {2, {65326, 65358}}, {2, {65327, 65359}}, {2, {65328, 65360}}, {2, {65329, 65361}}, {2, {65330, 65362}}, {2, {65331, 65363}}, {2, {65332, 65364}}, {2, {65333, 65365}}, {2, {65334, 65366}}, {2, {65335, 65367}}, {2, {65336, 65368}}, {2, {65337, 65369}}, {2, {65338, 65370}}, {0, {0}} }; // NOLINT
static const uint16_t kEcma262UnCanonicalizeTable1Size = 52;
static const int32_t kEcma262UnCanonicalizeTable1[104] = { 32545, 1, 32546, 5, 32547, 9, 32548, 13, 32549, 17, 32550, 21, 32551, 25, 32552, 29, 32553, 33, 32554, 37, 32555, 41, 32556, 45, 32557, 49, 32558, 53, 32559, 57, 32560, 61, 32561, 65, 32562, 69, 32563, 73, 32564, 77, 32565, 81, 32566, 85, 32567, 89, 32568, 93, 32569, 97, 32570, 101, 32577, 105, 32578, 109, 32579, 113, 32580, 117, 32581, 121, 32582, 125, 32583, 129, 32584, 133, 32585, 137, 32586, 141, 32587, 145, 32588, 149, 32589, 153, 32590, 157, 32591, 161, 32592, 165, 32593, 169, 32594, 173, 32595, 177, 32596, 181, 32597, 185, 32598, 189, 32599, 193, 32600, 197, 32601, 201, 32602, 205 }; // NOLINT
static const MultiCharacterSpecialCase<4> kEcma262UnCanonicalizeMultiStrings2[] = { {2, {66560, 66600}}, {2, {66561, 66601}}, {2, {66562, 66602}}, {2, {66563, 66603}}, {2, {66564, 66604}}, {2, {66565, 66605}}, {2, {66566, 66606}}, {2, {66567, 66607}}, {2, {66568, 66608}}, {2, {66569, 66609}}, {2, {66570, 66610}}, {2, {66571, 66611}}, {2, {66572, 66612}}, {2, {66573, 66613}}, {2, {66574, 66614}}, {2, {66575, 66615}}, {2, {66576, 66616}}, {2, {66577, 66617}}, {2, {66578, 66618}}, {2, {66579, 66619}}, {2, {66580, 66620}}, {2, {66581, 66621}}, {2, {66582, 66622}}, {2, {66583, 66623}}, {2, {66584, 66624}}, {2, {66585, 66625}}, {2, {66586, 66626}}, {2, {66587, 66627}}, {2, {66588, 66628}}, {2, {66589, 66629}}, {2, {66590, 66630}}, {2, {66591, 66631}}, {2, {66592, 66632}}, {2, {66593, 66633}}, {2, {66594, 66634}}, {2, {66595, 66635}}, {2, {66596, 66636}}, {2, {66597, 66637}}, {2, {66598, 66638}}, {2, {66599, 66639}}, {2, {66560, 66600}}, {2, {66561, 66601}}, {2, {66562, 66602}}, {2, {66563, 66603}}, {2, {66564, 66604}}, {2, {66565, 66605}}, {2, {66566, 66606}}, {2, {66567, 66607}}, {2, {66568, 66608}}, {2, {66569, 66609}}, {2, {66570, 66610}}, {2, {66571, 66611}}, {2, {66572, 66612}}, {2, {66573, 66613}}, {2, {66574, 66614}}, {2, {66575, 66615}}, {2, {66576, 66616}}, {2, {66577, 66617}}, {2, {66578, 66618}}, {2, {66579, 66619}}, {2, {66580, 66620}}, {2, {66581, 66621}}, {2, {66582, 66622}}, {2, {66583, 66623}}, {2, {66584, 66624}}, {2, {66585, 66625}}, {2, {66586, 66626}}, {2, {66587, 66627}}, {2, {66588, 66628}}, {2, {66589, 66629}}, {2, {66590, 66630}}, {2, {66591, 66631}}, {2, {66592, 66632}}, {2, {66593, 66633}}, {2, {66594, 66634}}, {2, {66595, 66635}}, {2, {66596, 66636}}, {2, {66597, 66637}}, {2, {66598, 66638}}, {2, {66599, 66639}}, {0, {0}} }; // NOLINT
static const uint16_t kEcma262UnCanonicalizeTable2Size = 80;
static const int32_t kEcma262UnCanonicalizeTable2[160] = { 1024, 1, 1025, 5, 1026, 9, 1027, 13, 1028, 17, 1029, 21, 1030, 25, 1031, 29, 1032, 33, 1033, 37, 1034, 41, 1035, 45, 1036, 49, 1037, 53, 1038, 57, 1039, 61, 1040, 65, 1041, 69, 1042, 73, 1043, 77, 1044, 81, 1045, 85, 1046, 89, 1047, 93, 1048, 97, 1049, 101, 1050, 105, 1051, 109, 1052, 113, 1053, 117, 1054, 121, 1055, 125, 1056, 129, 1057, 133, 1058, 137, 1059, 141, 1060, 145, 1061, 149, 1062, 153, 1063, 157, 1064, 161, 1065, 165, 1066, 169, 1067, 173, 1068, 177, 1069, 181, 1070, 185, 1071, 189, 1072, 193, 1073, 197, 1074, 201, 1075, 205, 1076, 209, 1077, 213, 1078, 217, 1079, 221, 1080, 225, 1081, 229, 1082, 233, 1083, 237, 1084, 241, 1085, 245, 1086, 249, 1087, 253, 1088, 257, 1089, 261, 1090, 265, 1091, 269, 1092, 273, 1093, 277, 1094, 281, 1095, 285, 1096, 289, 1097, 293, 1098, 297, 1099, 301, 1100, 305, 1101, 309, 1102, 313, 1103, 317 }; // NOLINT
int Ecma262UnCanonicalize::Convert(uchar c,
                      uchar n,
                      uchar* result,
                      bool* allow_caching_ptr) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupMapping(kEcma262UnCanonicalizeTable0,
                                     kEcma262UnCanonicalizeTable0Size,
                                     kEcma262UnCanonicalizeMultiStrings0,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 1: return LookupMapping(kEcma262UnCanonicalizeTable1,
                                     kEcma262UnCanonicalizeTable1Size,
                                     kEcma262UnCanonicalizeMultiStrings1,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 2: return LookupMapping(kEcma262UnCanonicalizeTable2,
                                     kEcma262UnCanonicalizeTable2Size,
                                     kEcma262UnCanonicalizeMultiStrings2,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    default: return 0;
  }
}

static const MultiCharacterSpecialCase<1> kCanonicalizationRangeMultiStrings0[] = { {0, {0}} }; // NOLINT
static const uint16_t kCanonicalizationRangeTable0Size = 1831;
static const int32_t kCanonicalizationRangeTable0[3662] = { 0, 67109124, 1073741825, 0, 64, 0, 65, 67108708, 1073741890, -260, 90, -260, 91, 67108524, 1073741916, -364, 96, -364, 97, 67108580, 1073741922, -388, 122, -388, 123, 67108604, 1073741948, -492, 180, -492, 181, 67108144, 182, 67108176, 1073742007, -728, 191, -728, 192, 67108188, 1073742017, -768, 214, -768, 215, 67108008, 216, 67108028, 1073742041, -864, 222, -864, 223, 67107976, 224, 67108060, 1073742049, -896, 246, -896, 247, 67107880, 248, 67107900, 1073742073, -992, 254, -992, 255, 67107848, 256, 67107844, 257, 67107840, 258, 67107836, 259, 67107832, 260, 67107828, 261, 67107824, 262, 67107820, 263, 67107816, 264, 67107812, 265, 67107808, 266, 67107804, 267, 67107800, 268, 67107796, 269, 67107792, 270, 67107788, 271, 67107784, 272, 67107780, 273, 67107776, 274, 67107772, 275, 67107768, 276, 67107764, 277, 67107760, 278, 67107756, 279, 67107752, 280, 67107748, 281, 67107744, 282, 67107740, 283, 67107736, 284, 67107732, 285, 67107728, 286, 67107724, 287, 67107720, 288, 67107716, 289, 67107712, 290, 67107708, 291, 67107704, 292, 67107700, 293, 67107696, 294, 67107692, 295, 67107688, 296, 67107684, 297, 67107680, 298, 67107676, 299, 67107672, 300, 67107668, 301, 67107664, 302, 67107660, 1073742127, 67107656, 304, 67107656, 305, -1216, 306, 67107644, 307, 67107640, 308, 67107636, 309, 67107632, 310, 67107628, 311, 67107624, 312, 67107620, 313, 67107616, 314, 67107612, 315, 67107608, 316, 67107604, 317, 67107600, 318, 67107596, 319, 67107592, 320, 67107588, 321, 67107584, 322, 67107580, 323, 67107576, 324, 67107572, 325, 67107568, 326, 67107564, 327, 67107560, 328, 67107556, 329, 67107552, 330, 67107548, 331, 67107544, 332, 67107540, 333, 67107536, 334, 67107532, 335, 67107528, 336, 67107524, 337, 67107520, 338, 67107516, 339, 67107512, 340, 67107508, 341, 67107504, 342, 67107500, 343, 67107496, 344, 67107492, 345, 67107488, 346, 67107484, 347, 67107480, 348, 67107476, 349, 67107472, 350, 67107468, 351, 67107464, 352, 67107460, 353, 67107456, 354, 67107452, 355, 67107448, 356, 67107444, 357, 67107440, 358, 67107436, 359, 67107432, 360, 67107428, 361, 67107424, 362, 67107420, 363, 67107416, 364, 67107412, 365, 67107408, 366, 67107404, 367, 67107400, 368, 67107396, 369, 67107392, 370, 67107388, 371, 67107384, 372, 67107380, 373, 67107376, 374, 67107372, 375, 67107368, 376, 67107364, 377, 67107360, 378, 67107356, 379, 67107352, 380, 67107348, 381, 67107344, 382, 67107340, 383, 67107336, 384, 67107332, 385, 67107328, 386, 67107324, 387, 67107320, 388, 67107316, 389, 67107312, 390, 67107308, 391, 67107304, 1073742216, 67107300, 393, 67107300, 394, -1572, 395, 67107288, 396, 67107284, 397, 67107280, 398, 67107276, 399, 67107272, 400, 67107268, 401, 67107264, 402, 67107260, 403, 67107256, 404, 67107252, 405, 67107248, 406, 67107244, 407, 67107240, 408, 67107236, 409, 67107232, 410, 67107228, 411, 67107224, 412, 67107220, 413, 67107216, 414, 67107212, 415, 67107208, 416, 67107204, 417, 67107200, 418, 67107196, 419, 67107192, 420, 67107188, 421, 67107184, 422, 67107180, 423, 67107176, 424, 67107172, 1073742249, 67107168, 426, 67107168, 427, -1704, 428, 67107156, 429, 67107152, 430, 67107148, 431, 67107144, 1073742256, 67107140, 433, 67107140, 434, -1732, 435, 67107128, 436, 67107124, 437, 67107120, 438, 67107116, 439, 67107112, 440, 67107108, 1073742265, 67107104, 442, 67107104, 443, -1768, 444, 67107092, 445, 67107088, 446, 67107084, 447, 67107080, 448, 67107088, 1073742273, -1792, 451, -1792, 452, 67107060, 453, 67107056, 454, 67107052, 455, 67107048, 456, 67107044, 457, 67107040, 458, 67107036, 459, 67107032, 460, 67107028, 461, 67107024, 462, 67107020, 463, 67107016, 464, 67107012, 465, 67107008, 466, 67107004, 467, 67107000, 468, 67106996, 469, 67106992, 470, 67106988, 471, 67106984, 472, 67106980, 473, 67106976, 474, 67106972, 475, 67106968, 476, 67106964, 477, 67106960, 478, 67106956, 479, 67106952, 480, 67106948, 481, 67106944, 482, 67106940, 483, 67106936, 484, 67106932, 485, 67106928, 486, 67106924, 487, 67106920, 488, 67106916, 489, 67106912, 490, 67106908, 491, 67106904, 492, 67106900, 493, 67106896, 494, 67106892, 495, 67106888, 496, 67106884, 497, 67106880, 498, 67106876, 499, 67106872, 500, 67106868, 501, 67106864, 502, 67106860, 503, 67106856, 504, 67106852, 505, 67106848, 506, 67106844, 507, 67106840, 508, 67106836, 509, 67106832, 510, 67106828, 511, 67106824, 512, 67106820, 513, 67106816, 514, 67106812, 515, 67106808, 516, 67106804, 517, 67106800, 518, 67106796, 519, 67106792, 520, 67106788, 521, 67106784, 522, 67106780, 523, 67106776, 524, 67106772, 525, 67106768, 526, 67106764, 527, 67106760, 528, 67106756, 529, 67106752, 530, 67106748, 531, 67106744, 532, 67106740, 533, 67106736, 534, 67106732, 535, 67106728, 536, 67106724, 537, 67106720, 538, 67106716, 539, 67106712, 540, 67106708, 541, 67106704, 542, 67106700, 543, 67106696, 544, 67106692, 545, 67106688, 546, 67106684, 547, 67106680, 548, 67106676, 549, 67106672, 550, 67106668, 551, 67106664, 552, 67106660, 553, 67106656, 554, 67106652, 555, 67106648, 556, 67106644, 557, 67106640, 558, 67106636, 559, 67106632, 560, 67106628, 561, 67106624, 562, 67106620, 563, 67106616, 564, 67106632, 1073742389, -2256, 569, -2256, 570, 67106588, 571, 67106584, 572, 67106580, 573, 67106576, 1073742398, 67106572, 575, 67106572, 576, -2300, 577, 67106560, 578, 67106556, 579, 67106552, 580, 67106548, 581, 67106544, 582, 67106540, 583, 67106536, 584, 67106532, 585, 67106528, 586, 67106524, 587, 67106520, 588, 67106516, 589, 67106512, 590, 67106508, 591, 67106504, 592, 67106508, 1073742417, -2368, 594, -2368, 595, 67106488, 596, 67106484, 1073742421, 67106480, 598, 67106480, 599, -2392, 600, 67106468, 601, 67106464, 602, 67106460, 603, 67106456, 604, 67106464, 1073742429, -2416, 607, -2416, 1073742432, 67106436, 609, 67106436, 610, -2436, 611, 67106424, 612, 67106432, 1073742437, -2448, 615, -2448, 616, 67106404, 617, 67106400, 618, 67106396, 619, 67106392, 620, 67106396, 1073742445, -2480, 622, -2480, 1073742447, 67106376, 624, 67106376, 625, -2496, 1073742450, 67106364, 627, 67106364, 628, -2508, 629, 67106352, 630, 67106372, 1073742455, -2520, 636, -2520, 1073742461, 67106320, 638, 67106320, 639, -2552, 1073742464, 67106308, 641, 67106308, 642, -2564, 643, 67106296, 644, 67106304, 1073742469, -2576, 647, -2576, 648, 67106276, 1073742473, 67106272, 650, 67106272, 651, -2600, 652, 67106260, 653, 67106272, 1073742478, -2612, 657, -2612, 658, 67106236, 659, 67106940, 1073742484, -2636, 836, -2636, 837, 67105520, 838, 67105724, 1073742663, -3352, 879, -3352, 1073742708, -3352, 885, -3352, 890, -3352, 891, 67105312, 1073742716, -3564, 893, -3564, 894, 67105320, 1073742724, -3576, 901, -3576, 902, 67105260, 903, 67105256, 904, 67105260, 1073742729, -3616, 906, -3616, 908, 67105236, 910, 67105232, 911, -3640, 912, 67105220, 913, 67105216, 1073742738, 67105212, 915, 67105212, 916, -3660, 1073742741, 67105200, 918, 67105200, 919, -3672, 920, 67105188, 921, 67105184, 922, 67105180, 923, 67105176, 924, 67105172, 925, 67105176, 1073742750, -3700, 927, -3700, 928, 67105156, 929, 67105152, 1073742755, 67105144, 932, 67105144, 933, -3728, 934, 67105132, 935, 67105144, 1073742760, -3740, 939, -3740, 940, 67105108, 941, 67105112, 1073742766, -3764, 943, -3764, 944, 67105092, 945, 67105088, 1073742770, 67105084, 947, 67105084, 948, -3788, 1073742773, 67105072, 950, 67105072, 951, -3800, 952, 67105060, 953, 67105056, 954, 67105052, 955, 67105048, 956, 67105044, 957, 67105048, 1073742782, -3828, 959, -3828, 960, 67105028, 961, 67105024, 962, 67105020, 1073742787, 67105016, 964, 67105016, 965, -3856, 966, 67105004, 967, 67105016, 1073742792, -3868, 971, -3868, 1073742796, 67104980, 973, 67104980, 974, -3892, 976, 67104964, 977, 67104960, 978, 67104964, 1073742803, -3912, 980, -3912, 981, 67104944, 982, 67104940, 983, 67104936, 984, 67104932, 985, 67104928, 986, 67104924, 987, 67104920, 988, 67104916, 989, 67104912, 990, 67104908, 991, 67104904, 992, 67104900, 993, 67104896, 994, 67104892, 995, 67104888, 996, 67104884, 997, 67104880, 998, 67104876, 999, 67104872, 1000, 67104868, 1001, 67104864, 1002, 67104860, 1003, 67104856, 1004, 67104852, 1005, 67104848, 1006, 67104844, 1007, 67104840, 1008, 67104836, 1009, 67104832, 1073742834, 67104828, 1011, 67104828, 1012, -4044, 1013, 67104816, 1014, 67104812, 1015, 67104808, 1016, 67104804, 1017, 67104800, 1018, 67104796, 1019, 67104792, 1020, 67104788, 1021, 67104792, 1073742846, -4084, 1023, -4084, 1024, 67104832, 1073742849, -4096, 1039, -4096, 1040, 67104832, 1073742865, -4160, 1071, -4160, 1072, 67104704, 1073742897, -4288, 1103, -4288, 1104, 67104512, 1073742929, -4416, 1119, -4416, 1120, 67104388, 1121, 67104384, 1122, 67104380, 1123, 67104376, 1124, 67104372, 1125, 67104368, 1126, 67104364, 1127, 67104360, 1128, 67104356, 1129, 67104352, 1130, 67104348, 1131, 67104344, 1132, 67104340, 1133, 67104336, 1134, 67104332, 1135, 67104328, 1136, 67104324, 1137, 67104320, 1138, 67104316, 1139, 67104312, 1140, 67104308, 1141, 67104304, 1142, 67104300, 1143, 67104296, 1144, 67104292, 1145, 67104288, 1146, 67104284, 1147, 67104280, 1148, 67104276, 1149, 67104272, 1150, 67104268, 1151, 67104264, 1152, 67104260, 1153, 67104256, 1154, 67104280, 1073742979, -4616, 1158, -4616, 1073742984, -4616, 1161, -4616, 1162, 67104220, 1163, 67104216, 1164, 67104212, 1165, 67104208, 1166, 67104204, 1167, 67104200, 1168, 67104196, 1169, 67104192, 1170, 67104188, 1171, 67104184, 1172, 67104180, 1173, 67104176, 1174, 67104172, 1175, 67104168, 1176, 67104164, 1177, 67104160, 1178, 67104156, 1179, 67104152, 1180, 67104148, 1181, 67104144, 1182, 67104140, 1183, 67104136, 1184, 67104132, 1185, 67104128, 1186, 67104124, 1187, 67104120, 1188, 67104116, 1189, 67104112, 1190, 67104108, 1191, 67104104, 1192, 67104100, 1193, 67104096, 1194, 67104092, 1195, 67104088, 1196, 67104084, 1197, 67104080, 1198, 67104076, 1199, 67104072, 1200, 67104068, 1201, 67104064, 1202, 67104060, 1203, 67104056, 1204, 67104052, 1205, 67104048, 1206, 67104044, 1207, 67104040, 1208, 67104036, 1209, 67104032, 1210, 67104028, 1211, 67104024, 1212, 67104020, 1213, 67104016, 1214, 67104012, 1215, 67104008, 1216, 67104004, 1217, 67104000, 1218, 67103996, 1219, 67103992, 1220, 67103988, 1221, 67103984, 1222, 67103980, 1223, 67103976, 1224, 67103972, 1225, 67103968, 1226, 67103964, 1227, 67103960, 1228, 67103956, 1229, 67103952, 1230, 67103948, 1231, 67103944, 1232, 67103940, 1233, 67103936, 1234, 67103932, 1235, 67103928, 1236, 67103924, 1237, 67103920, 1238, 67103916, 1239, 67103912, 1240, 67103908, 1241, 67103904, 1242, 67103900, 1243, 67103896, 1244, 67103892, 1245, 67103888, 1246, 67103884, 1247, 67103880, 1248, 67103876, 1249, 67103872, 1250, 67103868, 1251, 67103864, 1252, 67103860, 1253, 67103856, 1254, 67103852, 1255, 67103848, 1256, 67103844, 1257, 67103840, 1258, 67103836, 1259, 67103832, 1260, 67103828, 1261, 67103824, 1262, 67103820, 1263, 67103816, 1264, 67103812, 1265, 67103808, 1266, 67103804, 1267, 67103800, 1268, 67103796, 1269, 67103792, 1270, 67103788, 1271, 67103784, 1272, 67103780, 1273, 67103776, 1274, 67103772, 1275, 67103768, 1276, 67103764, 1277, 67103760, 1278, 67103756, 1279, 67103752, 1280, 67103748, 1281, 67103744, 1282, 67103740, 1283, 67103736, 1284, 67103732, 1285, 67103728, 1286, 67103724, 1287, 67103720, 1288, 67103716, 1289, 67103712, 1290, 67103708, 1291, 67103704, 1292, 67103700, 1293, 67103696, 1294, 67103692, 1295, 67103688, 1296, 67103684, 1297, 67103680, 1298, 67103676, 1299, 67103672, 1329, 67103700, 1073743154, -5316, 1366, -5316, 1073743193, -5468, 1375, -5468, 1377, 67103508, 1073743202, -5508, 1414, -5508, 1415, 67114568, 1073743241, -5660, 1418, -5660, 1073743249, -5660, 1479, -5660, 1073743312, -5660, 1514, -5660, 1073743344, -5660, 1524, -5660, 1073743360, -5660, 1539, -5660, 1073743371, -5660, 1557, -5660, 1563, -5660, 1073743390, -5660, 1567, -5660, 1073743393, -5660, 1594, -5660, 1073743424, -5660, 1630, -5660, 1073743456, -5660, 1805, -5660, 1073743631, -5660, 1866, -5660, 1073743693, -5660, 1901, -5660, 1073743744, -5660, 1969, -5660, 1073743808, -5660, 2042, -5660, 1073744129, -5660, 2361, -5660, 1073744188, -5660, 2381, -5660, 1073744208, -5660, 2388, -5660, 1073744216, -5660, 2416, -5660, 1073744251, -5660, 2431, -5660, 1073744257, -5660, 2435, -5660, 1073744261, -5660, 2444, -5660, 1073744271, -5660, 2448, -5660, 1073744275, -5660, 2472, -5660, 1073744298, -5660, 2480, -5660, 2482, -5660, 1073744310, -5660, 2489, -5660, 1073744316, -5660, 2500, -5660, 1073744327, -5660, 2504, -5660, 1073744331, -5660, 2510, -5660, 2519, -5660, 1073744348, -5660, 2525, -5660, 1073744351, -5660, 2531, -5660, 1073744358, -5660, 2554, -5660, 1073744385, -5660, 2563, -5660, 1073744389, -5660, 2570, -5660, 1073744399, -5660, 2576, -5660, 1073744403, -5660, 2600, -5660, 1073744426, -5660, 2608, -5660, 1073744434, -5660, 2611, -5660, 1073744437, -5660, 2614, -5660, 1073744440, -5660, 2617, -5660, 2620, -5660, 1073744446, -5660, 2626, -5660, 1073744455, -5660, 2632, -5660, 1073744459, -5660, 2637, -5660, 1073744473, -5660, 2652, -5660, 2654, -5660, 1073744486, -5660, 2676, -5660, 1073744513, -5660, 2691, -5660, 1073744517, -5660, 2701, -5660, 1073744527, -5660, 2705, -5660, 1073744531, -5660, 2728, -5660, 1073744554, -5660, 2736, -5660, 1073744562, -5660, 2739, -5660, 1073744565, -5660, 2745, -5660, 1073744572, -5660, 2757, -5660, 1073744583, -5660, 2761, -5660, 1073744587, -5660, 2765, -5660, 2768, -5660, 1073744608, -5660, 2787, -5660, 1073744614, -5660, 2799, -5660, 2801, -5660, 1073744641, -5660, 2819, -5660, 1073744645, -5660, 2828, -5660, 1073744655, -5660, 2832, -5660, 1073744659, -5660, 2856, -5660, 1073744682, -5660, 2864, -5660, 1073744690, -5660, 2867, -5660, 1073744693, -5660, 2873, -5660, 1073744700, -5660, 2883, -5660, 1073744711, -5660, 2888, -5660, 1073744715, -5660, 2893, -5660, 1073744726, -5660, 2903, -5660, 1073744732, -5660, 2909, -5660, 1073744735, -5660, 2913, -5660, 1073744742, -5660, 2929, -5660, 1073744770, -5660, 2947, -5660, 1073744773, -5660, 2954, -5660, 1073744782, -5660, 2960, -5660, 1073744786, -5660, 2965, -5660, 1073744793, -5660, 2970, -5660, 2972, -5660, 1073744798, -5660, 2975, -5660, 1073744803, -5660, 2980, -5660, 1073744808, -5660, 2986, -5660, 1073744814, -5660, 3001, -5660, 1073744830, -5660, 3010, -5660, 1073744838, -5660, 3016, -5660, 1073744842, -5660, 3021, -5660, 3031, -5660, 1073744870, -5660, 3066, -5660, 1073744897, -5660, 3075, -5660, 1073744901, -5660, 3084, -5660, 1073744910, -5660, 3088, -5660, 1073744914, -5660, 3112, -5660, 1073744938, -5660, 3123, -5660, 1073744949, -5660, 3129, -5660, 1073744958, -5660, 3140, -5660, 1073744966, -5660, 3144, -5660, 1073744970, -5660, 3149, -5660, 1073744981, -5660, 3158, -5660, 1073744992, -5660, 3169, -5660, 1073744998, -5660, 3183, -5660, 1073745026, -5660, 3203, -5660, 1073745029, -5660, 3212, -5660, 1073745038, -5660, 3216, -5660, 1073745042, -5660, 3240, -5660, 1073745066, -5660, 3251, -5660, 1073745077, -5660, 3257, -5660, 1073745084, -5660, 3268, -5660, 1073745094, -5660, 3272, -5660, 1073745098, -5660, 3277, -5660, 1073745109, -5660, 3286, -5660, 3294, -5660, 1073745120, -5660, 3299, -5660, 1073745126, -5660, 3311, -5660, 1073745137, -5660, 3314, -5660, 1073745154, -5660, 3331, -5660, 1073745157, -5660, 3340, -5660, 1073745166, -5660, 3344, -5660, 1073745170, -5660, 3368, -5660, 1073745194, -5660, 3385, -5660, 1073745214, -5660, 3395, -5660, 1073745222, -5660, 3400, -5660, 1073745226, -5660, 3405, -5660, 3415, -5660, 1073745248, -5660, 3425, -5660, 1073745254, -5660, 3439, -5660, 1073745282, -5660, 3459, -5660, 1073745285, -5660, 3478, -5660, 1073745306, -5660, 3505, -5660, 1073745331, -5660, 3515, -5660, 3517, -5660, 1073745344, -5660, 3526, -5660, 3530, -5660, 1073745359, -5660, 3540, -5660, 3542, -5660, 1073745368, -5660, 3551, -5660, 1073745394, -5660, 3572, -5660, 1073745409, -5660, 3642, -5660, 1073745471, -5660, 3675, -5660, 1073745537, -5660, 3714, -5660, 3716, -5660, 1073745543, -5660, 3720, -5660, 3722, -5660, 3725, -5660, 1073745556, -5660, 3735, -5660, 1073745561, -5660, 3743, -5660, 1073745569, -5660, 3747, -5660, 3749, -5660, 3751, -5660, 1073745578, -5660, 3755, -5660, 1073745581, -5660, 3769, -5660, 1073745595, -5660, 3773, -5660, 1073745600, -5660, 3780, -5660, 3782, -5660, 1073745608, -5660, 3789, -5660, 1073745616, -5660, 3801, -5660, 1073745628, -5660, 3805, -5660, 1073745664, -5660, 3911, -5660, 1073745737, -5660, 3946, -5660, 1073745777, -5660, 3979, -5660, 1073745808, -5660, 3991, -5660, 1073745817, -5660, 4028, -5660, 1073745854, -5660, 4044, -5660, 1073745871, -5660, 4049, -5660, 1073745920, -5660, 4129, -5660, 1073745955, -5660, 4135, -5660, 1073745961, -5660, 4138, -5660, 1073745964, -5660, 4146, -5660, 1073745974, -5660, 4153, -5660, 1073745984, -5660, 4185, -5660, 4256, 67091992, 1073746081, -17024, 4293, -17024, 1073746128, -17176, 4348, -17176, 1073746176, -17176, 4441, -17176, 1073746271, -17176, 4514, -17176, 1073746344, -17176, 4601, -17176, 1073746432, -17176, 4680, -17176, 1073746506, -17176, 4685, -17176, 1073746512, -17176, 4694, -17176, 4696, -17176, 1073746522, -17176, 4701, -17176, 1073746528, -17176, 4744, -17176, 1073746570, -17176, 4749, -17176, 1073746576, -17176, 4784, -17176, 1073746610, -17176, 4789, -17176, 1073746616, -17176, 4798, -17176, 4800, -17176, 1073746626, -17176, 4805, -17176, 1073746632, -17176, 4822, -17176, 1073746648, -17176, 4880, -17176, 1073746706, -17176, 4885, -17176, 1073746712, -17176, 4954, -17176, 1073746783, -17176, 4988, -17176, 1073746816, -17176, 5017, -17176, 1073746848, -17176, 5108, -17176, 1073746945, -17176, 5750, -17176, 1073747584, -17176, 5788, -17176, 1073747616, -17176, 5872, -17176, 1073747712, -17176, 5900, -17176, 1073747726, -17176, 5908, -17176, 1073747744, -17176, 5942, -17176, 1073747776, -17176, 5971, -17176, 1073747808, -17176, 5996, -17176, 1073747822, -17176, 6000, -17176, 1073747826, -17176, 6003, -17176, 1073747840, -17176, 6109, -17176, 1073747936, -17176, 6121, -17176, 1073747952, -17176, 6137, -17176, 1073747968, -17176, 6158, -17176, 1073747984, -17176, 6169, -17176, 1073748000, -17176, 6263, -17176, 1073748096, -17176, 6313, -17176, 1073748224, -17176, 6428, -17176, 1073748256, -17176, 6443, -17176, 1073748272, -17176, 6459, -17176, 6464, -17176, 1073748292, -17176, 6509, -17176, 1073748336, -17176, 6516, -17176, 1073748352, -17176, 6569, -17176, 1073748400, -17176, 6601, -17176, 1073748432, -17176, 6617, -17176, 1073748446, -17176, 6683, -17176, 1073748510, -17176, 6687, -17176, 1073748736, -17176, 6987, -17176, 1073748816, -17176, 7036, -17176, 1073749248, -17176, 7548, -17176, 7549, 67078672, 7550, 67079184, 1073749375, -30200, 7626, -30200, 1073749502, -30200, 7679, -30200, 7680, 67078148, 7681, 67078144, 7682, 67078140, 7683, 67078136, 7684, 67078132, 7685, 67078128, 7686, 67078124, 7687, 67078120, 7688, 67078116, 7689, 67078112, 7690, 67078108, 7691, 67078104, 7692, 67078100, 7693, 67078096, 7694, 67078092, 7695, 67078088, 7696, 67078084, 7697, 67078080, 7698, 67078076, 7699, 67078072, 7700, 67078068, 7701, 67078064, 7702, 67078060, 7703, 67078056, 7704, 67078052, 7705, 67078048, 7706, 67078044, 7707, 67078040, 7708, 67078036, 7709, 67078032, 7710, 67078028, 7711, 67078024, 7712, 67078020, 7713, 67078016, 7714, 67078012, 7715, 67078008, 7716, 67078004, 7717, 67078000, 7718, 67077996, 7719, 67077992, 7720, 67077988, 7721, 67077984, 7722, 67077980, 7723, 67077976, 7724, 67077972, 7725, 67077968, 7726, 67077964, 7727, 67077960, 7728, 67077956, 7729, 67077952, 7730, 67077948, 7731, 67077944, 7732, 67077940, 7733, 67077936, 7734, 67077932, 7735, 67077928, 7736, 67077924, 7737, 67077920, 7738, 67077916, 7739, 67077912, 7740, 67077908, 7741, 67077904, 7742, 67077900, 7743, 67077896, 7744, 67077892, 7745, 67077888, 7746, 67077884, 7747, 67077880, 7748, 67077876, 7749, 67077872, 7750, 67077868, 7751, 67077864, 7752, 67077860, 7753, 67077856, 7754, 67077852, 7755, 67077848, 7756, 67077844, 7757, 67077840, 7758, 67077836, 7759, 67077832, 7760, 67077828, 7761, 67077824, 7762, 67077820, 7763, 67077816, 7764, 67077812, 7765, 67077808, 7766, 67077804, 7767, 67077800, 7768, 67077796, 7769, 67077792, 7770, 67077788, 7771, 67077784, 7772, 67077780, 7773, 67077776, 7774, 67077772, 7775, 67077768, 7776, 67077764, 7777, 67077760, 7778, 67077756, 7779, 67077752, 7780, 67077748, 7781, 67077744, 7782, 67077740, 7783, 67077736, 7784, 67077732, 7785, 67077728, 7786, 67077724, 7787, 67077720, 7788, 67077716, 7789, 67077712, 7790, 67077708, 7791, 67077704, 7792, 67077700, 7793, 67077696, 7794, 67077692, 7795, 67077688, 7796, 67077684, 7797, 67077680, 7798, 67077676, 7799, 67077672, 7800, 67077668, 7801, 67077664, 7802, 67077660, 7803, 67077656, 7804, 67077652, 7805, 67077648, 7806, 67077644, 7807, 67077640, 7808, 67077636, 7809, 67077632, 7810, 67077628, 7811, 67077624, 7812, 67077620, 7813, 67077616, 7814, 67077612, 7815, 67077608, 7816, 67077604, 7817, 67077600, 7818, 67077596, 7819, 67077592, 7820, 67077588, 7821, 67077584, 7822, 67077580, 7823, 67077576, 7824, 67077572, 7825, 67077568, 7826, 67077564, 7827, 67077560, 7828, 67077556, 7829, 67077552, 7830, 67077564, 1073749655, -31320, 7834, -31320, 7835, 67077528, 7840, 67077508, 7841, 67077504, 7842, 67077500, 7843, 67077496, 7844, 67077492, 7845, 67077488, 7846, 67077484, 7847, 67077480, 7848, 67077476, 7849, 67077472, 7850, 67077468, 7851, 67077464, 7852, 67077460, 7853, 67077456, 7854, 67077452, 7855, 67077448, 7856, 67077444, 7857, 67077440, 7858, 67077436, 7859, 67077432, 7860, 67077428, 7861, 67077424, 7862, 67077420, 7863, 67077416, 7864, 67077412, 7865, 67077408, 7866, 67077404, 7867, 67077400, 7868, 67077396, 7869, 67077392, 7870, 67077388, 7871, 67077384, 7872, 67077380, 7873, 67077376, 7874, 67077372, 7875, 67077368, 7876, 67077364, 7877, 67077360, 7878, 67077356, 7879, 67077352, 7880, 67077348, 7881, 67077344, 7882, 67077340, 7883, 67077336, 7884, 67077332, 7885, 67077328, 7886, 67077324, 7887, 67077320, 7888, 67077316, 7889, 67077312, 7890, 67077308, 7891, 67077304, 7892, 67077300, 7893, 67077296, 7894, 67077292, 7895, 67077288, 7896, 67077284, 7897, 67077280, 7898, 67077276, 7899, 67077272, 7900, 67077268, 7901, 67077264, 7902, 67077260, 7903, 67077256, 7904, 67077252, 7905, 67077248, 7906, 67077244, 7907, 67077240, 7908, 67077236, 7909, 67077232, 7910, 67077228, 7911, 67077224, 7912, 67077220, 7913, 67077216, 7914, 67077212, 7915, 67077208, 7916, 67077204, 7917, 67077200, 7918, 67077196, 7919, 67077192, 7920, 67077188, 7921, 67077184, 7922, 67077180, 7923, 67077176, 7924, 67077172, 7925, 67077168, 7926, 67077164, 7927, 67077160, 7928, 67077156, 7929, 67077152, 7936, 67077152, 1073749761, -31744, 7943, -31744, 7944, 67077120, 1073749769, -31776, 7951, -31776, 7952, 67077080, 1073749777, -31808, 7957, -31808, 7960, 67077048, 1073749785, -31840, 7965, -31840, 7968, 67077024, 1073749793, -31872, 7975, -31872, 7976, 67076992, 1073749801, -31904, 7983, -31904, 7984, 67076960, 1073749809, -31936, 7991, -31936, 7992, 67076928, 1073749817, -31968, 7999, -31968, 8000, 67076888, 1073749825, -32000, 8005, -32000, 8008, 67076856, 1073749833, -32032, 8013, -32032, 8016, -32056, 8017, 67076800, 8018, 67076796, 8019, 67076792, 8020, 67076788, 8021, 67076784, 8022, 67076780, 8023, 67076776, 8025, 67076768, 8027, 67076760, 8029, 67076752, 8031, 67076744, 8032, 67076768, 1073749857, -32128, 8039, -32128, 8040, 67076736, 1073749865, -32160, 8047, -32160, 8048, 67076680, 8049, -32192, 8050, 67076680, 1073749875, -32200, 8053, -32200, 8054, 67076656, 8055, -32216, 8056, 67076648, 8057, -32224, 8058, 67076640, 8059, -32232, 8060, 67076632, 8061, -32240, 1073749888, -32248, 8111, -32248, 8112, 67076424, 8113, -32448, 8114, 67076432, 1073749939, -32456, 8116, -32456, 1073749942, -32456, 8119, -32456, 8120, 67076392, 8121, -32480, 8122, 67076384, 8123, -32488, 8124, 67076376, 8125, -32496, 8126, 67076364, 8127, 67076392, 1073749952, -32508, 8132, -32508, 1073749958, -32508, 8135, -32508, 8136, 67076336, 1073749961, -32544, 8139, -32544, 8140, 67076320, 1073749965, -32560, 8143, -32560, 8144, 67076296, 8145, -32576, 8146, 67076304, 8147, -32584, 1073749974, -32584, 8151, -32584, 8152, 67076264, 8153, -32608, 8154, 67076256, 8155, -32616, 1073749981, -32624, 8159, -32624, 8160, 67076232, 8161, -32640, 8162, 67076228, 1073749987, -32648, 8164, -32648, 1073749989, 67076208, 8166, 67076208, 8167, -32664, 8168, 67076200, 8169, -32672, 8170, 67076192, 8171, -32680, 8172, 67076180, 8173, 67076216, 1073749998, -32692, 8175, -32692, 1073750002, -32692, 8180, -32692, 1073750006, -32692, 8183, -32692, 8184, 67076136, 8185, -32736, 8186, 67076128, 8187, -32744, 8188, 67077352, 1073750013, -32752, 8190, -32752, 1073750016, -32752, 8291, -32752, 1073750122, -32752, 8305, -32752, 1073750132, -32752, 8334, -32752, 1073750160, -32752, 8340, -32752, 1073750176, -32752, 8373, -32752, 1073750224, -32752, 8431, -32752, 1073750272, -32752, 8497, -32752, 8498, 67074876, 8499, 67074976, 1073750324, -33996, 8525, -33996, 8526, 67074764, 1073750355, -34108, 8543, -34108, 8544, 67074752, 1073750369, -34176, 8559, -34176, 8560, 67074688, 1073750385, -34240, 8575, -34240, 8576, 67074572, 1073750401, -34304, 8578, -34304, 8579, 67074552, 8580, 67074548, 1073750416, -34324, 9191, -34324, 1073751040, -34324, 9254, -34324, 1073751104, -34324, 9290, -34324, 1073751136, -34324, 9397, -34324, 9398, 67071376, 1073751223, -37592, 9423, -37592, 9424, 67071272, 1073751249, -37696, 9449, -37696, 9450, 67078320, 1073751275, -37800, 9884, -37800, 1073751712, -37800, 9906, -37800, 1073751809, -37800, 9988, -37800, 1073751814, -37800, 9993, -37800, 1073751820, -37800, 10023, -37800, 1073751849, -37800, 10059, -37800, 10061, -37800, 1073751887, -37800, 10066, -37800, 10070, -37800, 1073751896, -37800, 10078, -37800, 1073751905, -37800, 10132, -37800, 1073751960, -37800, 10159, -37800, 1073751985, -37800, 10174, -37800, 1073752000, -37800, 10186, -37800, 1073752016, -37800, 10219, -37800, 1073752048, -37800, 11034, -37800, 1073752864, -37800, 11043, -37800, 11264, 67063996, 1073753089, -45056, 11310, -45056, 11312, 67063804, 1073753137, -45248, 11358, -45248, 11360, 67063428, 11361, 67063424, 11362, 67063420, 11363, 67063416, 11364, 67063412, 11365, 67063408, 11366, 67063404, 11367, 67063400, 11368, 67063396, 11369, 67063392, 11370, 67063388, 11371, 67063384, 11372, 67063380, 11380, -45492, 11381, 67063344, 11382, 67063340, 11383, 67063368, 11392, 67063300, 11393, 67063296, 11394, 67063292, 11395, 67063288, 11396, 67063284, 11397, 67063280, 11398, 67063276, 11399, 67063272, 11400, 67063268, 11401, 67063264, 11402, 67063260, 11403, 67063256, 11404, 67063252, 11405, 67063248, 11406, 67063244, 11407, 67063240, 11408, 67063236, 11409, 67063232, 11410, 67063228, 11411, 67063224, 11412, 67063220, 11413, 67063216, 11414, 67063212, 11415, 67063208, 11416, 67063204, 11417, 67063200, 11418, 67063196, 11419, 67063192, 11420, 67063188, 11421, 67063184, 11422, 67063180, 11423, 67063176, 11424, 67063172, 11425, 67063168, 11426, 67063164, 11427, 67063160, 11428, 67063156, 11429, 67063152, 11430, 67063148, 11431, 67063144, 11432, 67063140, 11433, 67063136, 11434, 67063132, 11435, 67063128, 11436, 67063124, 11437, 67063120, 11438, 67063116, 11439, 67063112, 11440, 67063108, 11441, 67063104, 11442, 67063100, 11443, 67063096, 11444, 67063092, 11445, 67063088, 11446, 67063084, 11447, 67063080, 11448, 67063076, 11449, 67063072, 11450, 67063068, 11451, 67063064, 11452, 67063060, 11453, 67063056, 11454, 67063052, 11455, 67063048, 11456, 67063044, 11457, 67063040, 11458, 67063036, 11459, 67063032, 11460, 67063028, 11461, 67063024, 11462, 67063020, 11463, 67063016, 11464, 67063012, 11465, 67063008, 11466, 67063004, 11467, 67063000, 11468, 67062996, 11469, 67062992, 11470, 67062988, 11471, 67062984, 11472, 67062980, 11473, 67062976, 11474, 67062972, 11475, 67062968, 11476, 67062964, 11477, 67062960, 11478, 67062956, 11479, 67062952, 11480, 67062948, 11481, 67062944, 11482, 67062940, 11483, 67062936, 11484, 67062932, 11485, 67062928, 11486, 67062924, 11487, 67062920, 11488, 67062916, 11489, 67062912, 11490, 67062908, 11491, 67062904, 11492, 67063008, 1073753317, -45968, 11498, -45968, 1073753337, -45968, 11519, -45968, 11520, 67062936, 1073753345, -46080, 11557, -46080, 1073753392, -46232, 11621, -46232, 11631, -46232, 1073753472, -46232, 11670, -46232, 1073753504, -46232, 11686, -46232, 1073753512, -46232, 11694, -46232, 1073753520, -46232, 11702, -46232, 1073753528, -46232, 11710, -46232, 1073753536, -46232, 11718, -46232, 1073753544, -46232, 11726, -46232, 1073753552, -46232, 11734, -46232, 1073753560, -46232, 11742, -46232, 1073753600, -46232, 11799, -46232, 1073753628, -46232, 11805, -46232, 1073753728, -46232, 11929, -46232, 1073753755, -46232, 12019, -46232, 1073753856, -46232, 12245, -46232, 1073754096, -46232, 12283, -46232, 1073754112, -46232, 12351, -46232, 1073754177, -46232, 12438, -46232, 1073754265, -46232, 12543, -46232, 1073754373, -46232, 12588, -46232, 1073754417, -46232, 12686, -46232, 1073754512, -46232, 12727, -46232, 1073754560, -46232, 12751, -46232, 1073754608, -46232, 12830, -46232, 1073754656, -46232, 12867, -46232, 1073754704, -46232, 13054, -46232, 1073754880, -46232, 19893, -46232, 1073761728, -46232, 32767, -46232 }; // NOLINT
static const MultiCharacterSpecialCase<1> kCanonicalizationRangeMultiStrings1[] = { {0, {0}} }; // NOLINT
static const uint16_t kCanonicalizationRangeTable1Size = 88;
static const int32_t kCanonicalizationRangeTable1[176] = { 1073741824, -46232, 8123, -46232, 1073750016, -46232, 9356, -46232, 1073751184, -46232, 9414, -46232, 1073751808, -46232, 10010, -46232, 1073751840, -46232, 10017, -46232, 1073752064, -46232, 10283, -46232, 1073752128, -46232, 10359, -46232, 1073753088, -46232, 22435, -46232, 1073764352, -46232, 31277, -46232, 1073773104, -46232, 31338, -46232, 1073773168, -46232, 31449, -46232, 1073773312, -46232, 31494, -46232, 1073773331, -46232, 31511, -46232, 1073773341, -46232, 31542, -46232, 1073773368, -46232, 31548, -46232, 31550, -46232, 1073773376, -46232, 31553, -46232, 1073773379, -46232, 31556, -46232, 1073773382, -46232, 31665, -46232, 1073773523, -46232, 32063, -46232, 1073773904, -46232, 32143, -46232, 1073773970, -46232, 32199, -46232, 1073774064, -46232, 32253, -46232, 1073774080, -46232, 32281, -46232, 1073774112, -46232, 32291, -46232, 1073774128, -46232, 32338, -46232, 1073774164, -46232, 32358, -46232, 1073774184, -46232, 32363, -46232, 1073774192, -46232, 32372, -46232, 1073774198, -46232, 32508, -46232, 32511, -46232, 1073774337, -46232, 32544, -46232, 32545, 66847716, 1073774370, -261252, 32570, -261252, 32571, 66847532, 1073774396, -261356, 32576, -261356, 32577, 66847588, 1073774402, -261380, 32602, -261380, 32603, 66848040, 1073774428, -261484, 32702, -261484, 1073774530, -261484, 32711, -261484, 1073774538, -261484, 32719, -261484, 1073774546, -261484, 32727, -261484, 1073774554, -261484, 32732, -261484, 1073774560, -261484, 32742, -261484, 1073774568, -261484, 32750, -261484, 1073774585, -261484, 32765, -261484 }; // NOLINT
int CanonicalizationRange::Convert(uchar c,
                      uchar n,
                      uchar* result,
                      bool* allow_caching_ptr) {
  int chunk_index = c >> 15;
  switch (chunk_index) {
    case 0: return LookupMapping(kCanonicalizationRangeTable0,
                                     kCanonicalizationRangeTable0Size,
                                     kCanonicalizationRangeMultiStrings0,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    case 1: return LookupMapping(kCanonicalizationRangeTable1,
                                     kCanonicalizationRangeTable1Size,
                                     kCanonicalizationRangeMultiStrings1,
                                     c,
                                     n,
                                     result,
                                     allow_caching_ptr);
    default: return 0;
  }
}


uchar UnicodeData::kMaxCodePoint = 1114109;

int UnicodeData::GetByteCount() {
  return 0 + (sizeof(uint16_t) * kUppercaseTable0Size) + (sizeof(uint16_t) * kUppercaseTable1Size) + (sizeof(uint16_t) * kUppercaseTable2Size) + (sizeof(uint16_t) * kUppercaseTable3Size) + (sizeof(uint16_t) * kLowercaseTable0Size) + (sizeof(uint16_t) * kLowercaseTable1Size) + (sizeof(uint16_t) * kLowercaseTable2Size) + (sizeof(uint16_t) * kLowercaseTable3Size) + (sizeof(uint16_t) * kLetterTable0Size) + (sizeof(uint16_t) * kLetterTable1Size) + (sizeof(uint16_t) * kLetterTable2Size) + (sizeof(uint16_t) * kLetterTable3Size) + (sizeof(uint16_t) * kLetterTable4Size) + (sizeof(uint16_t) * kLetterTable5Size) + (sizeof(uint16_t) * kSpaceTable0Size) + (sizeof(uint16_t) * kNumberTable0Size) + (sizeof(uint16_t) * kNumberTable1Size) + (sizeof(uint16_t) * kNumberTable2Size) + (sizeof(uint16_t) * kNumberTable3Size) + (sizeof(uint16_t) * kWhiteSpaceTable0Size) + (sizeof(uint16_t) * kLineTerminatorTable0Size) + (sizeof(uint16_t) * kCombiningMarkTable0Size) + (sizeof(uint16_t) * kCombiningMarkTable1Size) + (sizeof(uint16_t) * kCombiningMarkTable2Size) + (sizeof(uint16_t) * kCombiningMarkTable3Size) + (sizeof(uint16_t) * kCombiningMarkTable28Size) + (sizeof(uint16_t) * kConnectorPunctuationTable0Size) + (sizeof(uint16_t) * kConnectorPunctuationTable1Size) + (sizeof(uint16_t) * kToLowercaseTable0Size) + (sizeof(uint16_t) * kToLowercaseTable1Size) + (sizeof(uint16_t) * kToLowercaseTable2Size) + (sizeof(uint16_t) * kToUppercaseTable0Size) + (sizeof(uint16_t) * kToUppercaseTable1Size) + (sizeof(uint16_t) * kToUppercaseTable2Size) + (sizeof(uint16_t) * kEcma262CanonicalizeTable0Size) + (sizeof(uint16_t) * kEcma262CanonicalizeTable1Size) + (sizeof(uint16_t) * kEcma262CanonicalizeTable2Size) + (sizeof(uint16_t) * kEcma262UnCanonicalizeTable0Size) + (sizeof(uint16_t) * kEcma262UnCanonicalizeTable1Size) + (sizeof(uint16_t) * kEcma262UnCanonicalizeTable2Size) + (sizeof(uint16_t) * kCanonicalizationRangeTable0Size) + (sizeof(uint16_t) * kCanonicalizationRangeTable1Size); // NOLINT
}

}  // namespace unicode
