// Copyright 2012 the V8 project authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef V8_HEAP_INCREMENTAL_MARKING_H_
#define V8_HEAP_INCREMENTAL_MARKING_H_

#include <cstdint>
#include <optional>

#include "src/base/hashing.h"
#include "src/base/logging.h"
#include "src/base/platform/mutex.h"
#include "src/base/platform/time.h"
#include "src/common/globals.h"
#include "src/heap/heap.h"
#include "src/heap/incremental-marking-job.h"
#include "src/heap/mark-compact.h"
#include "src/heap/mutable-page-metadata.h"
#include "src/tasks/cancelable-task.h"

namespace v8 {
namespace internal {

class HeapObject;
class MarkBit;
class Map;
class Object;
class PagedSpace;

// Describes in which context IncrementalMarking::Step() is used in. This
// information is used when marking finishes and for marking progress
// heuristics.
enum class StepOrigin {
  // The caller of Step() is not allowed to complete marking right away. A task
  // is scheduled to complete the GC. When the task isn't
  // run soon enough, the stack guard mechanism will be used.
  kV8,

  // The caller of Step() will complete marking by running the GC right
  // afterwards.
  kTask
};

constexpr const char* ToString(StepOrigin step_origin) {
  switch (step_origin) {
    case StepOrigin::kV8:
      return "V8";
    case StepOrigin::kTask:
      return "task";
  }
}

class V8_EXPORT_PRIVATE IncrementalMarking final {
 public:
  IncrementalMarking(Heap* heap, WeakObjects* weak_objects);

  IncrementalMarking(const IncrementalMarking&) = delete;
  IncrementalMarking& operator=(const IncrementalMarking&) = delete;

  MarkingMode marking_mode() const { return marking_mode_; }

  bool IsMinorMarking() const {
    return marking_mode_ == MarkingMode::kMinorMarking;
  }
  bool IsMajorMarking() const {
    return marking_mode_ == MarkingMode::kMajorMarking;
  }

  bool IsStopped() const { return !IsMarking(); }
  bool IsMarking() const { return marking_mode_ != MarkingMode::kNoMarking; }
  bool IsMajorMarkingComplete() const {
    return IsMajorMarking() && ShouldFinalize();
  }

  bool MajorCollectionRequested() const {
    return major_collection_requested_via_stack_guard_;
  }

  // Checks whether incremental marking is safe to be started and whether it
  // should be started.
  bool CanAndShouldBeStarted() const;
  void Start(GarbageCollector garbage_collector,
             GarbageCollectionReason gc_reason);
  // Returns true if incremental marking was running and false otherwise.
  bool Stop();

  // Performs incremental marking step and finalizes marking if complete.
  void AdvanceAndFinalizeIfComplete();

  // Performs incremental marking step and finalizes marking if the stack guard
  // was already armed. If marking is complete but the stack guard wasn't armed
  // yet, a finalization task is scheduled.
  void AdvanceAndFinalizeIfNecessary();

  // Performs incremental marking step and schedules job for finalization if
  // marking completes.
  void AdvanceOnAllocation();

  bool IsCompacting() { return IsMajorMarking() && is_compacting_; }

  Heap* heap() const { return heap_; }
  Isolate* isolate() const;

  IncrementalMarkingJob* incremental_marking_job() const {
    return incremental_marking_job_.get();
  }

  bool black_allocation() { return black_allocation_; }

  bool IsBelowActivationThresholds() const;

  void MarkBlackBackground(Tagged<HeapObject> obj, int object_size);

  void MarkRootsForTesting();

  // Performs incremental marking step for unit tests.
  void AdvanceForTesting(v8::base::TimeDelta max_duration,
                         size_t max_bytes_to_mark = SIZE_MAX);

  uint64_t current_trace_id() const { return current_trace_id_.value(); }

  std::shared_ptr<::heap::base::IncrementalMarkingSchedule> schedule() {
    return schedule_;
  }

 private:
  class Observer final : public AllocationObserver {
   public:
    Observer(IncrementalMarking* incremental_marking, intptr_t step_size);
    ~Observer() override = default;
    void Step(int bytes_allocated, Address, size_t) override;

   private:
    IncrementalMarking* const incremental_marking_;
  };

  void StartMarkingMajor();
  void StartMarkingMinor();

  // Checks whether incremental marking is safe to be started.
  bool CanBeStarted() const;

  void StartBlackAllocation();
  void PauseBlackAllocation();
  void FinishBlackAllocation();

  void StartPointerTableBlackAllocation();
  void StopPointerTableBlackAllocation();

  void MarkRoots();

  void PublishWriteBarrierWorklists();

  // Fetches marked byte counters from the concurrent marker.
  void FetchBytesMarkedConcurrently();
  size_t GetScheduledBytes(StepOrigin step_origin);

  bool ShouldFinalize() const;

  bool ShouldWaitForTask();
  bool TryInitializeTaskTimeout();

  // Returns the actual used time and actually marked bytes.
  std::pair<v8::base::TimeDelta, size_t> CppHeapStep(
      v8::base::TimeDelta max_duration,
      std::optional<size_t> marked_bytes_limit, StepOrigin step_origin);

  void Step(v8::base::TimeDelta max_duration, size_t max_bytes_to_process,
            StepOrigin step_origin);

  size_t OldGenerationSizeOfObjects() const;

  MarkingState* marking_state() { return marking_state_; }
  MarkingWorklists::Local* local_marking_worklists() const {
    return current_local_marking_worklists_;
  }

  Heap* const heap_;
  MarkCompactCollector* const major_collector_;
  MinorMarkSweepCollector* const minor_collector_;
  WeakObjects* weak_objects_;
  MarkingWorklists::Local* current_local_marking_worklists_ = nullptr;
  MarkingState* const marking_state_;
  v8::base::TimeTicks start_time_;
  size_t main_thread_marked_bytes_ = 0;
  // A sample of concurrent_marking()->TotalMarkedBytes() at the last
  // incremental marking step.
  size_t bytes_marked_concurrently_ = 0;
  MarkingMode marking_mode_ = MarkingMode::kNoMarking;

  bool is_compacting_ = false;
  bool black_allocation_ = false;
  bool completion_task_scheduled_ = false;
  v8::base::TimeTicks completion_task_timeout_;
  bool major_collection_requested_via_stack_guard_ = false;
  std::unique_ptr<IncrementalMarkingJob> incremental_marking_job_;
  Observer new_generation_observer_;
  Observer old_generation_observer_;
  base::Mutex background_live_bytes_mutex_;
  std::unordered_map<MutablePageMetadata*, intptr_t,
                     base::hash<MutablePageMetadata*>>
      background_live_bytes_;
  std::shared_ptr<::heap::base::IncrementalMarkingSchedule> schedule_;
  std::optional<uint64_t> current_trace_id_;

  friend class IncrementalMarkingJob;
};

}  // namespace internal
}  // namespace v8

#endif  // V8_HEAP_INCREMENTAL_MARKING_H_
